import React, { useState, useEffect } from 'react';
import PropTypes from 'prop-types';
import { styled } from '@mui/material/styles';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  TextField,
  Button,
  Box,
  Typography,
  Alert,
  CircularProgress,
  IconButton,
  Divider,
  Chip,
  Grid,
  Tooltip,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  FormControlLabel,
  Checkbox,
  FormHelperText
} from '@mui/material';
import Table from '@mui/material/Table';
import TableBody from '@mui/material/TableBody';
import TableCell from '@mui/material/TableCell';
import TableContainer from '@mui/material/TableContainer';
import TableHead from '@mui/material/TableHead';
import TableRow from '@mui/material/TableRow';
import Paper from '@mui/material/Paper';
import CloseIcon from '@mui/icons-material/Close';
import RocketLaunchIcon from '@mui/icons-material/RocketLaunch';
import AddIcon from '@mui/icons-material/Add';
import DeleteIcon from '@mui/icons-material/Delete';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import SmartToyIcon from '@mui/icons-material/SmartToy';
import axios from 'axios';
import { BASE_URL } from '../../const';
import yaml from 'js-yaml';

import { SYSTEM_VARIABLES, SYSTEM_VARIABLE_KEYS } from './constants';

// Styled components
const StyledDialog = styled(Dialog)(({ theme }) => ({
  '& .MuiDialog-paper': {
    borderRadius: 12,
    backgroundColor: theme.palette.background.paper,
    backgroundImage: 'linear-gradient(rgba(0, 192, 127, 0.03), rgba(0, 0, 0, 0.03))',
    minWidth: 700,
    maxHeight: '90vh',
  },
}));

const StyledDialogTitle = styled(DialogTitle)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  justifyContent: 'space-between',
  paddingBottom: theme.spacing(1),
}));

const TitleText = styled(Typography)(({ theme }) => ({
  fontSize: '1.5rem',
  fontWeight: 600,
  color: theme.palette.primary.main,
}));

const StyledDialogContent = styled(DialogContent)(({ theme }) => ({
  paddingTop: theme.spacing(2),
}));

const FormSection = styled(Box)(({ theme }) => ({
  marginBottom: theme.spacing(3),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  fontSize: '1rem',
  fontWeight: 600,
  marginBottom: theme.spacing(1.5),
  color: theme.palette.text.primary,
  display: 'flex',
  alignItems: 'center',
  gap: theme.spacing(1),
}));

const InfoBox = styled(Box)(({ theme }) => ({
  backgroundColor: 'rgba(0, 192, 127, 0.05)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: 8,
  padding: theme.spacing(2),
  marginBottom: theme.spacing(2),
}));

const InfoRow = styled(Box)(({ theme }) => ({
  display: 'flex',
  justifyContent: 'space-between',
  marginBottom: theme.spacing(0.5),
  '&:last-child': {
    marginBottom: 0,
  },
}));

const StyledTextField = styled(TextField)(({ theme }) => ({
  '& .MuiOutlinedInput-root': {
    '&:hover fieldset': {
      borderColor: theme.palette.primary.main,
    },
  },
}));

const DeployButton = styled(Button)(({ theme }) => ({
  textTransform: 'none',
  fontWeight: 500,
  backgroundColor: theme.palette.primary.main,
  color: theme.palette.primary.contrastText,
  '&:hover': {
    backgroundColor: theme.palette.primary.dark,
  },
  '&:disabled': {
    backgroundColor: theme.palette.action.disabledBackground,
  },
}));

const EnvVarRow = styled(Box)(({ theme }) => ({
  display: 'flex',
  gap: theme.spacing(1),
  alignItems: 'flex-start',
  marginBottom: theme.spacing(1),
}));

const SYSTEM_VARIABLE_SET = new Set(SYSTEM_VARIABLE_KEYS);

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
}));

const VariableCode = styled(Typography)(({ theme }) => ({
  fontFamily: theme.typography.fontFamilyMono || 'Roboto Mono, monospace',
  fontSize: '0.85rem',
  padding: theme.spacing(0.5, 1),
  borderRadius: theme.spacing(0.5),
  backgroundColor: theme.palette.action.hover,
  display: 'inline-block',
}));

const AppDeployModal = ({ template, open, onClose, onSuccess = () => {}, ephemeralDefault = false }) => {
  const [deploymentName, setDeploymentName] = useState(
    `${template.name}-${Date.now().toString(36)}`
  );
  const [instances, setInstances] = useState(1);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [envVars, setEnvVars] = useState([]);
  const [portMappings, setPortMappings] = useState([]);
  const [ephemeralSession, setEphemeralSession] = useState(Boolean(ephemeralDefault));
  const [ephemeralForced, setEphemeralForced] = useState(false);
  
  // Model preference states
  // Use nullish coalescing to preserve null values (don't coerce null to 'any')
  const [preferredModelType, setPreferredModelType] = useState(template.preferred_model_type ?? 'any');
  const [failIfModelTypeUnavailable, setFailIfModelTypeUnavailable] = useState(template.fail_if_model_type_unavailable || false);
  const [preferredModelName, setPreferredModelName] = useState(template.preferred_model_name || '');
  const [failIfModelNameUnavailable, setFailIfModelNameUnavailable] = useState(template.fail_if_model_name_unavailable || false);

  const systemVariableRows = React.useMemo(() => {
    return SYSTEM_VARIABLES.map((variable) => {
      let displayValue = variable.placeholder || '';

      if (variable.key === 'KAMIWAZA_APP_NAME') {
        displayValue = deploymentName || variable.placeholder;
      }

      return {
        ...variable,
        displayValue,
      };
    });
  }, [deploymentName]);

  useEffect(() => {
    // Initialize environment variables from template defaults
    const initialEnvVars = [];
    
    // Add template defaults first
    if (template.env_defaults) {
      const defaultVars = Object.entries(template.env_defaults)
        .filter(([key]) => key !== 'OPENAI_BASE_URI')
        .filter(([key]) => !SYSTEM_VARIABLE_SET.has(key))
        .map(([key, value]) => ({
          key,
          value,
          isDefault: true,
          source: 'template'
        }));
      initialEnvVars.push(...defaultVars);
    }

    setEnvVars(initialEnvVars);

    // Extract port mappings from compose file
    if (template.compose_yml) {
      try {
        const compose = yaml.load(template.compose_yml);
        const extractedPorts = [];
        
        if (compose.services) {
          Object.values(compose.services).forEach(service => {
            if (service.ports) {
              service.ports.forEach((port, index) => {
                const portStr = String(port);
                let containerPort;
                
                // Handle different port formats: "8080", "8080:8080", "0.0.0.0:8080:8080"
                if (portStr.includes(':')) {
                  const parts = portStr.split(':');
                  containerPort = parts[parts.length - 1];
                } else {
                  containerPort = portStr;
                }
                
                extractedPorts.push({
                  container_port: parseInt(containerPort),
                  port_name: index === 0 ? 'primary' : `port-${index + 1}`,
                  is_primary: index === 0
                });
              });
            }
          });
        }
        
        setPortMappings(extractedPorts.slice(0, 3)); // Limit to 3 ports
      } catch (err) {
        console.error('Error parsing compose file:', err);
      }
    }
  }, [template]);

  useEffect(() => {
    if (open) {
      setEphemeralSession(Boolean(ephemeralDefault));
    }
  }, [open, ephemeralDefault, template]);

  // Fetch ephemeral forced configuration on mount
  useEffect(() => {
    axios.get(`${BASE_URL}/apps/config/ephemeral_forced`)
      .then(res => {
        setEphemeralForced(res.data.ephemeral_forced);
        if (res.data.ephemeral_forced) {
          setEphemeralSession(true);
        }
      })
      .catch(() => {
        // Ignore errors, use default (false)
      });
  }, []);

  const handleAddEnvVar = () => {
    setEnvVars([...envVars, { key: '', value: '', isDefault: false, source: 'user' }]);
  };

  const handleRemoveEnvVar = (index) => {
    setEnvVars(envVars.filter((_, i) => i !== index));
  };

  const handleEnvVarChange = (index, field, value) => {
    const updated = [...envVars];
    updated[index] = { 
      ...updated[index], 
      [field]: value, 
      isDefault: false,  // Mark as modified when user changes it
      source: updated[index].source || 'user'  // Preserve source or mark as user
    };
    setEnvVars(updated);
  };

  // Helper function to format error messages
  const formatErrorMessage = (error) => {
    if (!error) return 'An unknown error occurred';

    // If error is a string, return it directly
    if (typeof error === 'string') return error;

    // If error is an array (Pydantic validation errors)
    if (Array.isArray(error)) {
      return error.map(err => {
        if (typeof err === 'string') return err;
        if (err.msg) return `${err.loc ? err.loc.join('.') + ': ' : ''}${err.msg}`;
        return JSON.stringify(err);
      }).join('; ');
    }

    // If error is an object with a message property
    if (error.msg) return error.msg;
    if (error.message) return error.message;

    // Fallback: try to stringify
    try {
      return JSON.stringify(error);
    } catch {
      return 'Failed to parse error message';
    }
  };

  const handleDeploy = async () => {
    setLoading(true);
    setError(null);

    try {
      // Convert env vars array to object
      const envVarsObject = {};
      envVars.forEach(({ key, value }) => {
        if (key && key.trim()) {
          envVarsObject[key] = value;
        }
      });

      // Prepare port mappings for deployment
      const deploymentPortMappings = portMappings.map(pm => ({
        container_port: pm.container_port,
        port_name: pm.port_name,
        is_primary: pm.is_primary
      }));

      const deploymentData = {
        name: deploymentName,
        template_id: template.id,
        starting_copies: instances,
        min_copies: 1,
        max_copies: instances,
        env_vars: Object.keys(envVarsObject).length > 0 ? envVarsObject : null,
        port_mappings: deploymentPortMappings.length > 0 ? deploymentPortMappings : null,
        // Add model preferences
        preferred_model_type: preferredModelType,
        fail_if_model_type_unavailable: failIfModelTypeUnavailable,
        preferred_model_name: preferredModelName || null,
        fail_if_model_name_unavailable: failIfModelNameUnavailable,
        is_ephemeral_session: ephemeralSession
      };

      const response = await axios.post(`${BASE_URL}/apps/deploy_app`, deploymentData);

      if (response.status === 200) {
        onSuccess();
      }
    } catch (err) {
      console.error('Error deploying app:', err);
      const errorDetail = err.response?.data?.detail;
      setError(formatErrorMessage(errorDetail) || 'Failed to deploy application');
    } finally {
      setLoading(false);
    }
  };

  const handleClose = () => {
    if (!loading) {
      setError(null);
      onClose();
    }
  };

  return (
    <StyledDialog open={open} onClose={handleClose} maxWidth="md" fullWidth>
      <StyledDialogTitle>
        <TitleText>Deploy Application</TitleText>
        <IconButton onClick={handleClose} disabled={loading}>
          <CloseIcon />
        </IconButton>
      </StyledDialogTitle>

      <Divider />

      <StyledDialogContent>
        {error && (
          <Alert severity="error" sx={{ mb: 2 }}>
            {error}
          </Alert>
        )}

        <FormSection>
          <SectionTitle>Application Template</SectionTitle>
          <InfoBox>
            <InfoRow>
              <Typography variant="body2" color="text.secondary">Name:</Typography>
              <Typography variant="body2" fontWeight={500}>
                {template.name.replace(/-/g, ' ').replace(/\b\w/g, l => l.toUpperCase())}
              </Typography>
            </InfoRow>
            <InfoRow>
              <Typography variant="body2" color="text.secondary">Version:</Typography>
              <Typography variant="body2" fontWeight={500}>
                {template.version || '1.0.0'}
              </Typography>
            </InfoRow>
            <InfoRow>
              <Typography variant="body2" color="text.secondary">Source:</Typography>
              <Typography variant="body2" fontWeight={500}>
                {template.source_type === 'kamiwaza' ? 'Official' : 'Community'}
              </Typography>
            </InfoRow>
          </InfoBox>
        </FormSection>

        <FormSection>
          <SectionTitle>Deployment Configuration</SectionTitle>
          
          <StyledTextField
            fullWidth
            label="Deployment Name"
            value={deploymentName}
            onChange={(e) => setDeploymentName(e.target.value)}
            margin="normal"
            required
            disabled={loading}
            helperText="A unique name for this deployment"
          />

          <StyledTextField
            fullWidth
            label="Number of Instances"
            type="number"
            value={instances}
            onChange={(e) => setInstances(Math.max(1, parseInt(e.target.value) || 1))}
            margin="normal"
            required
            disabled={loading}
            inputProps={{ min: 1, max: 10 }}
            helperText="Number of container instances to run"
          />

          <Box sx={{ mt: 2 }}>
            <FormControlLabel
              control={
                <Checkbox
                  checked={ephemeralSession}
                  onChange={(e) => setEphemeralSession(e.target.checked)}
                  disabled={loading || ephemeralForced}
                  sx={ephemeralForced ? {
                    '&.Mui-disabled': {
                      color: 'text.disabled',
                      opacity: 0.6,
                    },
                    '& .MuiSvgIcon-root': {
                      color: 'text.disabled',
                    }
                  } : {}}
                />
              }
              label={
                <Typography
                  component="span"
                  sx={ephemeralForced ? { color: 'text.disabled', fontStyle: 'italic' } : {}}
                >
                  Ephemeral session (purge on logout/timeout)
                  {ephemeralForced && ' (enforced)'}
                </Typography>
              }
            />
            <FormHelperText sx={{ ml: 1.5 }}>
              {ephemeralForced
                ? "Ephemeral mode is enforced by system configuration. All deployments will be automatically purged on logout."
                : "When enabled, this deployment and its data are destroyed automatically when you sign out or your session expires."}
            </FormHelperText>
          </Box>
        </FormSection>

        <FormSection>
          <SectionTitle>
            <SmartToyIcon sx={{ fontSize: '1.2rem' }} />
            Model Preferences
            <Tooltip title="Configure which AI model type this application should use">
              <InfoIcon />
            </Tooltip>
          </SectionTitle>
          
          <Grid container spacing={2}>
            <Grid item xs={12} md={6}>
              <FormControl fullWidth margin="normal">
                <InputLabel>Preferred Model Type</InputLabel>
                <Select
                  value={preferredModelType === null ? 'none' : preferredModelType}
                  onChange={(e) => setPreferredModelType(e.target.value === 'none' ? null : e.target.value)}
                  disabled={loading}
                  label="Preferred Model Type"
                >
                  <MenuItem value="none">None (No Model Required)</MenuItem>
                  <MenuItem value="any">Any Available Model</MenuItem>
                  <MenuItem value="large">Large (High Quality)</MenuItem>
                  <MenuItem value="fast">Fast (Low Latency)</MenuItem>
                  <MenuItem value="reasoning">Reasoning (CoT/Thinking)</MenuItem>
                  <MenuItem value="vl">Vision/Multimodal</MenuItem>
                </Select>
                <FormHelperText>
                  {preferredModelType === null
                    ? 'App will not connect to any model (use for static apps or apps with external APIs)'
                    : 'Select the type of model best suited for this application'}
                </FormHelperText>
              </FormControl>
            </Grid>
            
            <Grid item xs={12} md={6}>
              <Box sx={{ mt: 3 }}>
                <FormControlLabel
                  control={
                    <Checkbox
                      checked={failIfModelTypeUnavailable}
                      onChange={(e) => setFailIfModelTypeUnavailable(e.target.checked)}
                      disabled={loading || preferredModelType === 'any' || preferredModelType === null}
                    />
                  }
                  label="Fail if preferred type unavailable"
                />
              </Box>
            </Grid>
            
            <Grid item xs={12} md={6}>
              <StyledTextField
                fullWidth
                label="Preferred Model Name"
                value={preferredModelName}
                onChange={(e) => setPreferredModelName(e.target.value)}
                margin="normal"
                disabled={loading}
                placeholder="e.g., qwen, 72b, llama"
                helperText="Optional: Substring to match in model name"
              />
            </Grid>
            
            <Grid item xs={12} md={6}>
              <Box sx={{ mt: 3 }}>
                <FormControlLabel
                  control={
                    <Checkbox
                      checked={failIfModelNameUnavailable}
                      onChange={(e) => setFailIfModelNameUnavailable(e.target.checked)}
                      disabled={loading || !preferredModelName}
                    />
                  }
                  label="Fail if model name not found"
                />
              </Box>
            </Grid>
          </Grid>
          
          {(preferredModelType !== 'any' && preferredModelType !== null) || preferredModelName ? (
            <Alert severity="info" sx={{ mt: 2 }}>
              {preferredModelType !== 'any' && preferredModelType !== null && (
                <Typography variant="body2">
                  Will look for a <strong>{preferredModelType}</strong> model
                  {!failIfModelTypeUnavailable && ' (with fallback to other types if unavailable)'}.
                </Typography>
              )}
              {preferredModelName && (
                <Typography variant="body2">
                  Will look for models containing &quot;<strong>{preferredModelName}</strong>&quot;
                  {!failIfModelNameUnavailable && ' (will use any model if not found)'}.
                </Typography>
              )}
            </Alert>
          ) : null}
        </FormSection>

        <FormSection>
          <SectionTitle>
            Environment Variables
            <Tooltip title="Configure environment variables for your application">
              <InfoIcon />
            </Tooltip>
          </SectionTitle>
          
          <TableContainer component={Paper} variant="outlined" sx={{ mb: 2 }}>
            <Table size="small" aria-label="system variables">
              <TableHead>
                <TableRow>
                  <TableCell sx={{ width: '35%' }}>Variable</TableCell>
                  <TableCell>Value</TableCell>
                  <TableCell align="right" sx={{ width: '15%' }}>Info</TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                {systemVariableRows.map((variable) => (
                  <TableRow key={variable.key}>
                    <TableCell>
                      <VariableCode component="span">{variable.key}</VariableCode>
                    </TableCell>
                    <TableCell>
                      <Typography variant="body2" color="text.primary">
                        {variable.displayValue}
                      </Typography>
                    </TableCell>
                    <TableCell align="right">
                      <Tooltip title={variable.description} placement="left">
                        <IconButton size="small">
                          <InfoIcon fontSize="inherit" />
                        </IconButton>
                      </Tooltip>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </TableContainer>

          {envVars.map((envVar, index) => (
            <EnvVarRow key={index}>
              <StyledTextField
                label="Key"
                value={envVar.key}
                onChange={(e) => handleEnvVarChange(index, 'key', e.target.value)}
                disabled={loading || (envVar.isDefault && envVar.source === 'system')}
                size="small"
                sx={{ flex: 1 }}
                InputProps={{
                  endAdornment: envVar.isDefault && (
                    <Chip
                      label={envVar.source === 'system' ? 'System' : 'Template'} 
                      size="small" 
                      variant="outlined"
                      color={envVar.source === 'system' ? 'secondary' : 'primary'}
                    />
                  )
                }}
              />
              <StyledTextField
                label="Value"
                value={envVar.value}
                onChange={(e) => handleEnvVarChange(index, 'value', e.target.value)}
                disabled={loading}
                size="small"
                sx={{ flex: 1.5 }}
                helperText={envVar.source === 'system' && envVar.value.includes('host.docker.internal') ? 
                  'This value will be automatically set from deployed Kamiwaza models' : ''}
              />
              <IconButton
                onClick={() => handleRemoveEnvVar(index)}
                disabled={loading || (envVar.isDefault && envVar.source === 'system')}
                size="small"
                color="error"
                title={envVar.isDefault && envVar.source === 'system' ? 
                  'System environment variables cannot be removed' : 'Remove environment variable'}
              >
                <DeleteIcon />
              </IconButton>
            </EnvVarRow>
          ))}
          
          <Button
            startIcon={<AddIcon />}
            onClick={handleAddEnvVar}
            disabled={loading}
            size="small"
            sx={{ mt: 1 }}
          >
            Add Environment Variable
          </Button>
        </FormSection>
      </StyledDialogContent>

      <DialogActions sx={{ p: 2, pt: 0 }}>
        <Button onClick={handleClose} disabled={loading}>
          Cancel
        </Button>
        <DeployButton
          onClick={handleDeploy}
          disabled={loading || !deploymentName.trim()}
          startIcon={loading ? <CircularProgress size={20} /> : <RocketLaunchIcon />}
        >
          {loading ? 'Deploying...' : 'Deploy'}
        </DeployButton>
      </DialogActions>
    </StyledDialog>
  );
};

AppDeployModal.propTypes = {
  template: PropTypes.shape({
    id: PropTypes.string,
    name: PropTypes.string.isRequired,
    version: PropTypes.string,
    source_type: PropTypes.string,
    preferred_model_type: PropTypes.string,
    fail_if_model_type_unavailable: PropTypes.bool,
    preferred_model_name: PropTypes.string,
    fail_if_model_name_unavailable: PropTypes.bool,
    env_defaults: PropTypes.object,
    compose_yml: PropTypes.string,
  }).isRequired,
  open: PropTypes.bool.isRequired,
  onClose: PropTypes.func.isRequired,
  onSuccess: PropTypes.func,
  ephemeralDefault: PropTypes.bool,
};

AppDeployModal.defaultProps = {
  onSuccess: () => {},
  ephemeralDefault: false,
};

export default AppDeployModal;
