import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { styled } from '@mui/material/styles';
import { 
  Typography, 
  Box, 
  Paper, 
  List,
  ListItem,
  ListItemText,
  Divider,
  Tooltip,
  Chip,
  Avatar
} from '@mui/material';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import HistoryIcon from '@mui/icons-material/History';
import PersonIcon from '@mui/icons-material/Person';
import UpdateIcon from '@mui/icons-material/Update';
import { BASE_URL } from '../../const';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';

// Styled components matching the other updated pages
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 2),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const ActivityCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'hidden',
  marginBottom: theme.spacing(4),
}));

const ActivityList = styled(List)(({ theme }) => ({
  padding: 0,
}));

const ActivityItem = styled(ListItem)(({ theme }) => ({
  padding: theme.spacing(2, 3),
  borderBottom: '1px solid rgba(255, 255, 255, 0.05)',
  transition: 'background-color 0.2s',
  '&:last-child': {
    borderBottom: 'none',
  },
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.03)',
  },
}));

const ActivityContent = styled(ListItemText)(({ theme }) => ({
  margin: 0,
  '& .MuiListItemText-primary': {
    display: 'flex',
    alignItems: 'center',
    flexWrap: 'wrap',
    gap: theme.spacing(1),
    marginBottom: theme.spacing(0.5),
  },
  '& .MuiListItemText-secondary': {
    color: theme.palette.text.secondary,
  },
}));

const TimeChip = styled(Chip)(({ theme }) => ({
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  color: theme.palette.primary.main,
  fontWeight: 500,
  fontSize: '0.75rem',
  height: 24,
}));

const UserChip = styled(Chip)(({ theme }) => ({
  backgroundColor: 'rgba(30, 136, 229, 0.08)',
  color: theme.palette.secondary.main,
  fontWeight: 500,
  fontSize: '0.75rem',
  height: 24,
}));

const EmptyState = styled(Box)(({ theme }) => ({
  padding: theme.spacing(4),
  textAlign: 'center',
  color: theme.palette.text.secondary,
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  cursor: 'pointer',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.main,
  },
}));

function ActivityFeed() {
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [activities, setActivities] = useState([]);

  useEffect(() => {
    axios.get(`${BASE_URL}/activity/activities/`)
      .then(response => {
        setActivities(response.data);
        setLoading(false);
      })
      .catch(error => {
        setError(error.message);
        setLoading(false);
      });
  }, []);

  const formatDate = (dateString) => {
    const date = new Date(dateString);
    const today = new Date();
    if (date.toDateString() === today.toDateString()) {
      return `Today ${date.toLocaleTimeString()}`;
    } else {
      return `${date.toLocaleDateString()} ${date.toLocaleTimeString()}`;
    }
  };

  // Group activities by date
  const groupActivitiesByDate = () => {
    const grouped = {};
    
    activities.forEach(activity => {
      const date = new Date(activity.created_at);
      const dateString = date.toDateString();
      
      if (!grouped[dateString]) {
        grouped[dateString] = [];
      }
      
      grouped[dateString].push(activity);
    });
    
    return grouped;
  };

  const groupedActivities = groupActivitiesByDate();
  const activityDates = Object.keys(groupedActivities).sort((a, b) => new Date(b) - new Date(a)); // Sort dates descending

  return (
    <ContentContainer>
      <SectionTitle variant="h4">
        Recent Activities
        <Tooltip title="Log of recent actions performed by users in the system">
          <InfoIcon />
        </Tooltip>
      </SectionTitle>
      
      {loading && <Spinner />}
      {error && <ErrorComponent message={error} />}
      
      {!loading && !error && activities.length === 0 && (
        <EmptyState>
          <Typography variant="body1">No activities found</Typography>
        </EmptyState>
      )}
      
      {!loading && !error && activities.length > 0 && (
        <ActivityCard>
          <ActivityList>
            {activities.map((activity, index) => (
              <ActivityItem key={activity.id || index} alignItems="flex-start">
                <ActivityContent
                  primary={
                    <>
                      <TimeChip
                        icon={<UpdateIcon fontSize="small" />}
                        label={formatDate(activity.created_at)}
                        size="small"
                      />
                      {activity.user_id && (
                        <UserChip
                          avatar={<Avatar sx={{ width: 16, height: 16, fontSize: '0.75rem', bgcolor: 'secondary.main' }}>{activity.user_id.charAt(0).toUpperCase()}</Avatar>}
                          label={activity.user_id}
                          size="small"
                        />
                      )}
                      <Typography component="span">
                        {activity.action || `${activity.module} ${activity.apicall}`}
                      </Typography>
                    </>
                  }
                />
              </ActivityItem>
            ))}
          </ActivityList>
        </ActivityCard>
      )}
    </ContentContainer>
  );
}

export default ActivityFeed;
