# Frontend Testing Setup

Basic unit tests for React components using Jest and React Testing Library.

## Quick Start

```bash
npm test              # Run tests
npm run test:watch    # Watch mode
npm run test:coverage # With coverage
```

## Test Structure

Tests live in `__tests__` folders next to components:
```
components/common/
├── __tests__/
│   ├── Spinner.test.js
│   └── ConfirmModal.test.js
└── Spinner.js
```

## Helper Utilities

Import from `src/test-utils.js`:

```javascript
import { renderWithTheme, renderWithProviders, testData } from '../test-utils';

// Simple component testing
renderWithTheme(<MyComponent />);

// With routing/auth context (mocked)
renderWithProviders(<MyComponent />, {
  authValue: { user: testData.user, loading: false }
});

// Use test data
const mockModel = testData.model;
```

## Basic Patterns

```javascript
// Rendering
test('renders content', () => {
  const { getByText } = renderWithTheme(<Component title="Test" />);
  expect(getByText('Test')).toBeInTheDocument();
});

// User interactions
test('handles click', async () => {
  const user = userEvent.setup();
  const { getByRole } = renderWithTheme(<Button onClick={mockFn} />);
  await user.click(getByRole('button'));
  expect(mockFn).toHaveBeenCalled();
});

// Forms
test('submits form', async () => {
  const user = userEvent.setup();
  await user.type(getByLabelText('Name'), 'John');
  await user.click(getByRole('button', { name: /submit/i }));
  expect(mockSubmit).toHaveBeenCalledWith({ name: 'John' });
});
```

## What We Test

- User interactions (clicking, typing, navigation)
- Content rendering
- Error states and edge cases
- Accessibility basics
- Form submissions

## Current Status

We have basic tests for:
- ✅ Spinner component
- ✅ ErrorComponent 
- ✅ ConfirmModal

These serve as examples for testing other components throughout the app.