import { render, screen } from "@testing-library/react";
import userEvent from "@testing-library/user-event";
import React from "react";
import {
	MemoryRouter,
	Route,
	Routes,
	useLocation,
	useNavigate,
	useParams,
} from "react-router-dom";

describe("React Router v6 Core Functionality", () => {
	test("basic routing works with Routes and Route", () => {
		const Home = () => <div>Home Page</div>;
		const About = () => <div>About Page</div>;

		// Test home route
		const { unmount } = render(
			<MemoryRouter initialEntries={["/"]}>
				<Routes>
					<Route path="/" element={<Home />} />
					<Route path="/about" element={<About />} />
				</Routes>
			</MemoryRouter>,
		);

		expect(screen.getByText("Home Page")).toBeInTheDocument();
		expect(screen.queryByText("About Page")).not.toBeInTheDocument();

		unmount();

		// Test about route
		render(
			<MemoryRouter initialEntries={["/about"]}>
				<Routes>
					<Route path="/" element={<Home />} />
					<Route path="/about" element={<About />} />
				</Routes>
			</MemoryRouter>,
		);

		expect(screen.queryByText("Home Page")).not.toBeInTheDocument();
		expect(screen.getByText("About Page")).toBeInTheDocument();
	});

	test("useNavigate hook works", async () => {
		const user = userEvent.setup();

		const NavigationTest = () => {
			const navigate = useNavigate();
			return (
				<div>
					<button onClick={() => navigate("/target")}>Go to Target</button>
					<Routes>
						<Route path="/" element={<div>Start Page</div>} />
						<Route path="/target" element={<div>Target Page</div>} />
					</Routes>
				</div>
			);
		};

		render(
			<MemoryRouter initialEntries={["/"]}>
				<NavigationTest />
			</MemoryRouter>,
		);

		expect(screen.getByText("Start Page")).toBeInTheDocument();

		await user.click(screen.getByText("Go to Target"));

		expect(screen.getByText("Target Page")).toBeInTheDocument();
		expect(screen.queryByText("Start Page")).not.toBeInTheDocument();
	});

	test("useLocation hook works", () => {
		const LocationDisplay = () => {
			const location = useLocation();
			return <div>Current path: {location.pathname}</div>;
		};

		// Test first path
		const { unmount } = render(
			<MemoryRouter initialEntries={["/test"]}>
				<LocationDisplay />
			</MemoryRouter>,
		);

		expect(screen.getByText("Current path: /test")).toBeInTheDocument();

		unmount();

		// Test second path
		render(
			<MemoryRouter initialEntries={["/other"]}>
				<LocationDisplay />
			</MemoryRouter>,
		);

		expect(screen.getByText("Current path: /other")).toBeInTheDocument();
	});

	test("useParams hook works with route parameters", () => {
		const ParamsDisplay = () => {
			const params = useParams();
			return (
				<div>
					<div>ID: {params.id}</div>
					<div>Category: {params.category}</div>
				</div>
			);
		};

		render(
			<MemoryRouter initialEntries={["/products/electronics/123"]}>
				<Routes>
					<Route path="/products/:category/:id" element={<ParamsDisplay />} />
				</Routes>
			</MemoryRouter>,
		);

		expect(screen.getByText("ID: 123")).toBeInTheDocument();
		expect(screen.getByText("Category: electronics")).toBeInTheDocument();
	});

	test("nested routes work", () => {
		const Layout = () => (
			<div>
				<h1>Layout</h1>
				<Routes>
					<Route path="profile" element={<div>Profile Page</div>} />
					<Route path="settings" element={<div>Settings Page</div>} />
				</Routes>
			</div>
		);

		// Test profile route
		const { unmount } = render(
			<MemoryRouter initialEntries={["/user/profile"]}>
				<Routes>
					<Route path="/user/*" element={<Layout />} />
				</Routes>
			</MemoryRouter>,
		);

		expect(screen.getByText("Layout")).toBeInTheDocument();
		expect(screen.getByText("Profile Page")).toBeInTheDocument();

		unmount();

		// Test settings route
		render(
			<MemoryRouter initialEntries={["/user/settings"]}>
				<Routes>
					<Route path="/user/*" element={<Layout />} />
				</Routes>
			</MemoryRouter>,
		);

		expect(screen.getByText("Layout")).toBeInTheDocument();
		expect(screen.getByText("Settings Page")).toBeInTheDocument();
	});

	test("protected route pattern works", () => {
		// eslint-disable-next-line react/prop-types
		const ProtectedRoute = ({ children, isAuthenticated }) => {
			return isAuthenticated ? children : <div>Please log in</div>;
		};

		const SecretPage = () => <div>Secret Content</div>;

		// Not authenticated
		const { rerender } = render(
			<MemoryRouter initialEntries={["/secret"]}>
				<Routes>
					<Route
						path="/secret"
						element={
							<ProtectedRoute isAuthenticated={false}>
								<SecretPage />
							</ProtectedRoute>
						}
					/>
				</Routes>
			</MemoryRouter>,
		);

		expect(screen.getByText("Please log in")).toBeInTheDocument();
		expect(screen.queryByText("Secret Content")).not.toBeInTheDocument();

		// Authenticated
		rerender(
			<MemoryRouter initialEntries={["/secret"]}>
				<Routes>
					<Route
						path="/secret"
						element={
							<ProtectedRoute isAuthenticated={true}>
								<SecretPage />
							</ProtectedRoute>
						}
					/>
				</Routes>
			</MemoryRouter>,
		);

		expect(screen.getByText("Secret Content")).toBeInTheDocument();
		expect(screen.queryByText("Please log in")).not.toBeInTheDocument();
	});

	test("wildcard routes work", () => {
		const CatchAll = () => <div>404 - Not Found</div>;
		const Home = () => <div>Home</div>;

		render(
			<MemoryRouter initialEntries={["/non-existent-route"]}>
				<Routes>
					<Route path="/" element={<Home />} />
					<Route path="*" element={<CatchAll />} />
				</Routes>
			</MemoryRouter>,
		);

		expect(screen.getByText("404 - Not Found")).toBeInTheDocument();
	});

	test("multiple dynamic segments work", () => {
		const MultiParams = () => {
			const params = useParams();
			return (
				<div>
					{params.org}/{params.repo}/{params.branch}
				</div>
			);
		};

		render(
			<MemoryRouter initialEntries={["/github/facebook/react/main"]}>
				<Routes>
					<Route path="/github/:org/:repo/:branch" element={<MultiParams />} />
				</Routes>
			</MemoryRouter>,
		);

		expect(screen.getByText("facebook/react/main")).toBeInTheDocument();
	});
});
