import React, { useState, useEffect, useContext } from 'react';
import { ThemeProvider } from '@mui/material/styles';
import CssBaseline from '@mui/material/CssBaseline';
import { BrowserRouter as Router, Routes, Route, useLocation } from 'react-router-dom';
import Box from '@mui/material/Box';
import useMediaQuery from '@mui/material/useMediaQuery';
import Header from './components/common/Header';
import Sidebar from './components/common/Sidebar';
import ModelIndex from './components/models/ModelIndex';
import ModelDetail from './components/models/ModelDetail';
import { ModelFilesProvider } from './components/models/ModelFilesContext';
import ModelHubSearch from './components/models/ModelHubSearch';
import CatalogList from './components/dataset/DatasetList';
import ActivityFeed from './components/activity/ActivityFeed';
import DatasetDetails from './components/dataset/DatasetDetails';
import ClusterList from './components/cluster/ClusterList';
import ClusterHome from './components/cluster/ClusterHome';
import ClusterDetails from './components/cluster/ClusterDetails';
import LocationDetails from './components/cluster/LocationDetails';
import NodeDetails from './components/cluster/NodeDetails';
import HardwareDetails from './components/cluster/HardwareDetails';
import UserDetails from './components/cluster/UserDetails';
import UserList from './components/cluster/UserList';
import LocationList from './components/cluster/LocationList';
import NodeList from './components/cluster/NodeList';
import HardwareList from './components/cluster/HardwareList';
import RunningNodeDetail from './components/cluster/RunningNodeDetail';
import RunningNodeList from './components/cluster/RunningNodeList';
import Homepage from './components/home/Homepage';
import VectorDBHome from './components/vectordbs/VectorDbHome';
import VectorDBList from './components/vectordbs/VectorDBList';
import VectorDBDetails from './components/vectordbs/VectorDBDetails';
import AppGarden from './components/apps/AppGarden';
import ToolShed from './components/tool/ToolShed';
import UnifiedTemplateManager from './components/garden/UnifiedTemplateManager';
import News from './components/news/News.jsx';
import Settings from './components/settings/Settings';
import theme from './Theme';
import ModelConfigDetail from './components/models/ModelConfigDetail';
import DownloadStatusModal from './components/models/DownloadStatus';
import Cookies from 'js-cookie';
import { BASE_URL } from './const';
import { AuthProvider, AuthContext } from './context/AuthContext'; // Import AuthProvider & context
import { ServerInfoProvider } from './context/ServerInfoContext'; // <-- Add this import
import { SecurityProvider, useSecurity } from './context/SecurityContext'; // Import SecurityProvider
import { RoutingConfigProvider } from './context/RoutingConfigContext'; // Import RoutingConfigProvider
import { TourProvider } from './contexts/TourContext'; // Import TourProvider
import { ClusterProvider } from './contexts/ClusterContext'; // Import ClusterProvider
import Login from './components/auth/Login'; // Import Login component
import ProtectedRoute from './components/auth/ProtectedRoute'; // Import ProtectedRoute component
import AdminRoute from './components/auth/AdminRoute'; // Import AdminRoute component for admin-only routes
import { ConsentGate, ClassificationBanner, BANNER_HEIGHT } from './components/security'; // Import security components

// AppContent component to use location hooks
function AppContent() {
  // eslint-disable-next-line no-unused-vars
  const [hostResolvable, setHostResolvable] = useState(false);
  const [sidebarOpen, setSidebarOpen] = useState(true);
  // eslint-disable-next-line no-unused-vars
  const [advancedMode, setAdvancedMode] = useState(false);
  const isSmallScreen = useMediaQuery(theme.breakpoints.down('md'));
  const location = useLocation();
  const { user, loading } = useContext(AuthContext);
  const { shouldShowBanners } = useSecurity();

  useEffect(() => {
    // Check if advanced mode is enabled
    const savedMode = localStorage.getItem('advancedDeveloperMode');
    const isAdvanced = savedMode === 'true';
    setAdvancedMode(isAdvanced);

    // In novice mode, start with sidebar closed
    if (!isAdvanced) {
      setSidebarOpen(false);
    } else {
      // In advanced mode, follow normal responsive behavior
      if (isSmallScreen) {
        setSidebarOpen(false);
      } else {
        setSidebarOpen(true);
      }
    }
  }, [isSmallScreen]);

  useEffect(() => {
    if (loading || !user) {
      setHostResolvable(false);
      return;
    }

    let cancelled = false;

    const testHostResolvable = async () => {
      try {
        const hostnameResponse = await fetch(`${BASE_URL}/cluster/get_hostname`);
        let host = window.location.hostname;
        if (hostnameResponse.ok) {
          try {
            const body = await hostnameResponse.json();
            if (body && body.hostname) host = body.hostname;
          } catch (_) {
            // fall back to window hostname
          }
        }

        if (!host || cancelled) {
          if (!cancelled) {
            setHostResolvable(false);
            Cookies.set('hostnamesResolve', 'false', { expires: 1 });
          }
          return;
        }

        const img = new window.Image();
        img.src = `${window.location.protocol}//${host}/favicon.png`;

        img.onload = () => {
          if (cancelled) return;
          setHostResolvable(true);
          Cookies.set('hostnamesResolve', 'true', { expires: 1 });
        };

        img.onerror = () => {
          if (cancelled) return;
          setHostResolvable(false);
          Cookies.set('hostnamesResolve', 'false', { expires: 1 });
        };
      } catch (error) {
        if (cancelled) return;
        setHostResolvable(false);
        Cookies.set('hostnamesResolve', 'false', { expires: 1 });
      }
    };

    testHostResolvable();

    return () => {
      cancelled = true;
    };
  }, [loading, user]);

  const handleToggleSidebar = () => {
    setSidebarOpen(!sidebarOpen);
  };

  // Check if current path is login page
  const isLoginPage = location.pathname === '/login';

  return (
    <ModelFilesProvider>
      <Box sx={{
        display: 'flex',
        height: '100vh',
        // Constrain content to viewport minus banner heights
        paddingTop: shouldShowBanners ? `${BANNER_HEIGHT}px` : 0,
        paddingBottom: shouldShowBanners ? `${BANNER_HEIGHT}px` : 0,
        boxSizing: 'border-box',
        overflow: 'hidden', // Prevent outer scrolling - content scrolls inside
        bgcolor: 'background.default',
      }}>
        {/* Sidebar navigation component - only show when not on login page */}
        {!isLoginPage && <Sidebar open={sidebarOpen} onToggle={handleToggleSidebar} />}

        {/* Main content area */}
        <Box
          component="main"
          sx={{
            flexGrow: 1,
            overflow: 'auto', // Enable scrolling within content area
            transition: theme.transitions.create('margin', {
              easing: theme.transitions.easing.sharp,
              duration: theme.transitions.duration.leavingScreen,
            }),
            marginLeft: 0,
            width: '100%',
          }}
        >
          {/* Header at the top of the main content - conditionally rendered for login page */}
          {!isLoginPage && (
            <Header onToggleSidebar={handleToggleSidebar} sidebarOpen={sidebarOpen} />
          )}

          {/* Routes container with padding */}
          <Box sx={{ p: isLoginPage ? 0 : 3, mt: isLoginPage ? 0 : 8 }}>
            <Routes>
              {/* Login route (unprotected) */}
              <Route path="/login" element={<Login />} />

              {/* Protected routes */}
              <Route
                path="/"
                element={
                  <ProtectedRoute>
                    <Homepage />
                  </ProtectedRoute>
                }
              />
              <Route
                path="/apps"
                element={
                  <ProtectedRoute>
                    <AppGarden />
                  </ProtectedRoute>
                }
              />
              <Route
                path="/tool-shed"
                element={
                  <ProtectedRoute>
                    <ToolShed />
                  </ProtectedRoute>
                }
              />
              <Route
                path="/templates"
                element={
                  <ProtectedRoute>
                    <UnifiedTemplateManager />
                  </ProtectedRoute>
                }
              />
              <Route
                path="/models"
                element={
                  <ProtectedRoute>
                    <ModelIndex />
                  </ProtectedRoute>
                }
              />
              <Route
                path="/models/:model_id"
                element={
                  <ProtectedRoute>
                    <ModelDetail />
                  </ProtectedRoute>
                }
              />
              <Route
                path="/models/search"
                element={
                  <ProtectedRoute>
                    <ModelHubSearch />
                  </ProtectedRoute>
                }
              />
              <Route
                path="/model_configs/:model_config_id"
                element={
                  <ProtectedRoute>
                    <ModelConfigDetail />
                  </ProtectedRoute>
                }
              />
              {/* Catalog routes - accessible to all authenticated users */}
              <Route
                path="/catalog"
                element={
                  <ProtectedRoute>
                    <CatalogList />
                  </ProtectedRoute>
                }
              />
              <Route
                path="/catalog/datasets/*"
                element={
                  <ProtectedRoute>
                    <DatasetDetails />
                  </ProtectedRoute>
                }
              />
              <Route
                path="/activity"
                element={
                  <AdminRoute>
                    <ActivityFeed />
                  </AdminRoute>
                }
              />
              {/* Admin-only routes: Cluster */}
              <Route
                path="/cluster/locations/:location_id"
                element={
                  <AdminRoute>
                    <LocationDetails />
                  </AdminRoute>
                }
              />
              <Route
                path="/cluster/runningnodes/:node_id"
                element={
                  <AdminRoute>
                    <RunningNodeDetail />
                  </AdminRoute>
                }
              />
              <Route
                path="/cluster/runningnodes/"
                element={
                  <AdminRoute>
                    <RunningNodeList />
                  </AdminRoute>
                }
              />
              <Route
                path="/cluster/home"
                element={
                  <AdminRoute>
                    <ClusterHome />
                  </AdminRoute>
                }
              />
              <Route
                path="/cluster/:cluster_id"
                element={
                  <AdminRoute>
                    <ClusterDetails />
                  </AdminRoute>
                }
              />
              <Route
                path="/cluster/"
                element={
                  <AdminRoute>
                    <ClusterList />
                  </AdminRoute>
                }
              />
              <Route
                path="/cluster/node/:node_id"
                element={
                  <AdminRoute>
                    <NodeDetails />
                  </AdminRoute>
                }
              />
              <Route
                path="/cluster/hardware/:hardware_id"
                element={
                  <AdminRoute>
                    <HardwareDetails />
                  </AdminRoute>
                }
              />
              <Route
                path="/cluster/users/:user_id"
                element={
                  <AdminRoute>
                    <UserDetails />
                  </AdminRoute>
                }
              />
              <Route
                path="/cluster/users"
                element={
                  <AdminRoute>
                    <UserList />
                  </AdminRoute>
                }
              />
              <Route
                path="/cluster/locations"
                element={
                  <AdminRoute>
                    <LocationList />
                  </AdminRoute>
                }
              />
              <Route
                path="/cluster/nodes"
                element={
                  <AdminRoute>
                    <NodeList />
                  </AdminRoute>
                }
              />
              <Route
                path="/cluster/hardware"
                element={
                  <AdminRoute>
                    <HardwareList />
                  </AdminRoute>
                }
              />
              {/* Admin-only routes: VectorDBs */}
              <Route
                path="/vectordbs/home"
                element={
                  <AdminRoute>
                    <VectorDBHome />
                  </AdminRoute>
                }
              />
              <Route
                path="/vectordbs"
                element={
                  <AdminRoute>
                    <VectorDBList />
                  </AdminRoute>
                }
              />
              <Route
                path="/vectordbs/:vectordb_id"
                element={
                  <AdminRoute>
                    <VectorDBDetails />
                  </AdminRoute>
                }
              />
              {/* News route - accessible to all authenticated users */}
              <Route
                path="/news"
                element={
                  <ProtectedRoute>
                    <News />
                  </ProtectedRoute>
                }
              />
              <Route
                path="/settings"
                element={
                  <ProtectedRoute>
                    <Settings />
                  </ProtectedRoute>
                }
              />
            </Routes>
          </Box>
        </Box>

        {/* Download Status Modal - shown on all pages when downloads are active */}
        {!isLoginPage && <DownloadStatusModal />}
      </Box>
    </ModelFilesProvider>
  );
}

function App() {
  return (
    <ThemeProvider theme={theme}>
      <CssBaseline />
      <SecurityProvider>
        <ClassificationBanner position="top" />
        <ConsentGate>
          <AuthProvider>
            <ServerInfoProvider>
              <RoutingConfigProvider>
                <ClusterProvider>
                  <TourProvider>
                    <Router>
                      <AppContent />
                    </Router>
                  </TourProvider>
                </ClusterProvider>
              </RoutingConfigProvider>
            </ServerInfoProvider>
          </AuthProvider>
        </ConsentGate>
        <ClassificationBanner position="bottom" />
      </SecurityProvider>
    </ThemeProvider>
  );
}

export default App;
