#!/bin/bash
# set-node-version.sh - Ensure correct Node version for frontend
#
# Environment variables:
#   OFFLINE_MODE=true    - Skip nvm installation/updates for air-gapped environments
#   KAMIWAZA_COMMUNITY   - Community edition uses ~/.nvm, enterprise uses /opt/kamiwaza/nvm
#
# Usage:
#   source set-node-version.sh                    # Online mode (default)
#   OFFLINE_MODE=true source set-node-version.sh  # Offline/air-gapped mode

source "$(dirname "${BASH_SOURCE[0]}")/../set-kamiwaza-root.sh"
source "$(dirname "${BASH_SOURCE[0]}")/../common.sh"

# Determine env file location
if [[ -f "${KAMIWAZA_ROOT}/env.sh" ]]; then
    ENV_FILE="${KAMIWAZA_ROOT}/env.sh"
elif [[ -f "/etc/kamiwaza/env.sh" ]]; then
    ENV_FILE="/etc/kamiwaza/env.sh"
else
    echo "No env.sh file found in ${KAMIWAZA_ROOT} or /etc/kamiwaza"
    exit 1
fi

source "$ENV_FILE"

if ! command -v nvm &> /dev/null; then
    if [[ "${KAMIWAZA_COMMUNITY:-}" != "true" ]]; then
        # Enterprise edition: Set up NVM in /opt/kamiwaza/nvm
        NVM_INSTALL_DIR="/opt/kamiwaza/nvm"
    else
        # Community edition: Use standard home directory installation
        NVM_INSTALL_DIR="$HOME/.nvm"
    fi

    # Only try to install if nvm directory doesn't exist AND not in offline mode
    if [[ ! -d "${NVM_INSTALL_DIR}" ]] && [[ "${OFFLINE_MODE:-false}" != "true" ]]; then
        echo "=> nvm not found, attempting installation..."
        NVM_ENV_FILE="$ENV_FILE" bash ${NVM_INSTALL_DIR}/install.sh --no-use 2>/dev/null || {
            echo "=> nvm installation failed (possibly offline), continuing anyway..."
        }
    elif [[ "${OFFLINE_MODE:-false}" == "true" ]]; then
        echo "=> OFFLINE_MODE detected, skipping nvm installation/update"
    fi
fi

if ! command -v nvm &> /dev/null; then
    # Load nvm if it exists
    if [[ -s "${NVM_INSTALL_DIR}/nvm.sh" ]]; then
        # Disable nvm auto-update to prevent network calls
        export NVM_NO_UPDATE=1

        if [[ "${OFFLINE_MODE:-false}" == "true" ]]; then
            # Offline mode: suppress all network-related errors
            source ${NVM_INSTALL_DIR}/nvm.sh 2>/dev/null || true
        else
            # Online mode: normal loading
            source ${NVM_INSTALL_DIR}/nvm.sh
        fi
        [[ -s "${NVM_INSTALL_DIR}/bash_completion" ]] && source ${NVM_INSTALL_DIR}/bash_completion 2>/dev/null || true

    else
        if [[ "${OFFLINE_MODE:-false}" == "true" ]]; then
            echo "=> OFFLINE_MODE: nvm not available, using system node"
        else
            echo "=> nvm not found in ${NVM_INSTALL_DIR}, using system node"
        fi
    fi
fi

promote_nvm_node  # Called after nvm use to ensure proper PATH ordering
nvm use 22

# Verify we actually got Node 22
if ! verify_node_version; then
    echo "Failed to set Node 22 as active version" >&2
    exit 1
fi