#!/bin/bash
set -e

# use naive mode so etcd reads from file
export KAMIWAZA_INSTALL=true

# Parse command line arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --community)
            export KAMIWAZA_COMMUNITY=true
            export KAMIWAZA_SWARM_HEAD=true
            shift
            ;;
        --lite)
            export KAMIWAZA_LITE=true
            shift
            ;;
        --full)
            export KAMIWAZA_LITE=false
            shift
            ;;
        --head)
            export KAMIWAZA_SWARM_HEAD=true
            shift
            ;;
        --worker)
            shift
            ;;
        --head-ip=*)
            export KAMIWAZA_HEAD_IP="${1#*=}"
            # Auto-set KAMIWAZA_KEYCLOAK_HOST from KAMIWAZA_HEAD_IP if not already set
            if [[ -z "${KAMIWAZA_KEYCLOAK_HOST:-}" ]]; then
                export KAMIWAZA_KEYCLOAK_HOST="${KAMIWAZA_HEAD_IP}"
            fi
            shift
            ;;
        --without-exit)
            export KAMIWAZA_WITHOUT_EXIT=true
            shift
            ;;
        *)
            echo "Unknown option: $1"
            echo "Usage:"
            echo "  For community edition: $0 --community [--lite | --full] [--without-exit]"
            echo "  For head node: $0 --head [--full] [--without-exit]"
            echo "  For worker node: $0 --worker --head-ip=<head_ip> [--full] [--without-exit]"
            echo ""
            echo "Options:"
            echo "  --lite      Lite mode - only valid with --community"
            echo "  --full      Full mode - install full datahub and cockroachdb backends"
            echo "              (requires ~4-6GB of additional memory for community)"
            echo "  --without-exit   Do not exit after installation"
            exit 1
            ;;
    esac
done

# Validate lite/full flags
if [[ "${KAMIWAZA_LITE:-}" == "true" && "${KAMIWAZA_COMMUNITY:-}" != "true" ]]; then
    echo "Error: --lite flag is only valid with --community"
    exit 1
fi

# Set default for KAMIWAZA_LITE based on edition
if [[ "${KAMIWAZA_COMMUNITY:-}" == "true" && -z "${KAMIWAZA_LITE:-}" ]]; then
    export KAMIWAZA_LITE=true  # Default to lite for community
elif [[ "${KAMIWAZA_COMMUNITY:-}" != "true" ]]; then
    export KAMIWAZA_LITE=false  # Enterprise is always full
fi

if [ "${KAMIWAZA_SWARM_HEAD:-}" == "true" ]; then
    touch /tmp/kamiwazad.starting
else
    sudo mkdir -p /etc/kamiwaza/config
    sudo chown ${USER}:${USER} /etc/kamiwaza/config
    sudo chmod 700 /etc/kamiwaza/config
    echo '1' > /etc/kamiwaza/config/is_worker
fi

# Check for required parameters
if [[ "${KAMIWAZA_COMMUNITY:-}" != "true" && "${KAMIWAZA_SWARM_HEAD:-}" != "true" && -z "${KAMIWAZA_HEAD_IP:-}" ]]; then
    echo "Error: Must specify either --community, --head or --worker with --head-ip"
    echo "Usage:"
    echo "  For community edition: $0 --community [--lite | --full]"
    echo "  For head node: $0 --head [--full]"
    echo "  For worker node: $0 --worker --head-ip=<head_ip> [--full]"
    exit 1
fi

# Set up KAMIWAZA_ROOT from script directory
SCRIPT_DIR=$(dirname "${BASH_SOURCE[0]}")
source "${SCRIPT_DIR}/set-kamiwaza-root.sh"

cd "${SCRIPT_DIR}" || { echo "Failed to change to script directory"; exit 1; }

# Check if already set up
if [ -f ~/.kamiwaza-installed ]; then
    echo "Kamiwaza is already installed and configured."
    exit 0
fi

# Ensure Docker group membership (this is critical)
if ! groups | grep -q '\bdocker\b'; then
    sudo usermod -aG docker $USER || true
    echo "Added to docker group - you must log out and back in"
    if [[ "${KAMIWAZA_WITHOUT_EXIT:-}" == "true" ]]; then
        echo "Without exit mode detected: continuing without exit."
    else
        exit 1
    fi   
fi

# Then ensure the environment file exists and has proper permissions
if [[ "${KAMIWAZA_COMMUNITY:-}" != "true" ]]; then
    # Enterprise edition: env.sh goes in /etc/kamiwaza
    sudo mkdir -p /etc/kamiwaza
    sudo mkdir -p /opt/kamiwaza/logs
    sudo touch /etc/kamiwaza/env.sh
    sudo chown ${USER}:${USER} /etc/kamiwaza/env.sh /opt/kamiwaza/logs
    sudo chmod 640 /etc/kamiwaza/env.sh
else
    # Community edition: env.sh stays in KAMIWAZA_ROOT
    touch "${KAMIWAZA_ROOT}/env.sh"
    chmod 640 "${KAMIWAZA_ROOT}/env.sh"
fi

# Add near the start of first-boot.sh, after Docker group check
source common.sh
setup_environment

if [[ "${KAMIWAZA_COMMUNITY:-}" != "true" ]]; then
    if [ -f /etc/kamiwaza/env.sh ]; then
        source /etc/kamiwaza/env.sh
    else
        echo "No env.sh file found in /etc/kamiwaza - contact support@kamiwaza.ai - you should not use first-boot.sh without it."
        exit 1
    fi
else
    if [ -f "${KAMIWAZA_ROOT}/env.sh" ]; then
        source "${KAMIWAZA_ROOT}/env.sh"
    fi
fi

KAMIWAZA_SECURITY_DIR="${KAMIWAZA_SECURITY_DIR:-/etc/kamiwaza/security}"
FERNET_KEY_PATH="${FERNET_KEY_PATH:-${KAMIWAZA_SECURITY_DIR}/fernet.key}"
kw_resolve_fernet_identity

if [ ! -f "${KAMIWAZA_ROOT}/.venv/bin/python3" ]; then
    echo "No .venv/bin/python3 file found - contact support@kamiwaza.ai - you should not use first-boot.sh without it."
    exit 1
fi

# After the .venv/python check but before NVM setup
if [[ "${KAMIWAZA_SWARM_HEAD:-}" != "true" && "${KAMIWAZA_COMMUNITY:-}" != "true" ]]; then
    echo "Getting JWT keys from head node..."
    sudo mkdir -p ${KAMIWAZA_ROOT}/runtime
    # JWT keys now managed by Keycloak via JWKS - no local key files needed
    # Keys are fetched dynamically from Keycloak's JWKS endpoint
    
    echo "Getting fernet key from head node..."
    sudo mkdir -p "${KAMIWAZA_SECURITY_DIR}"
    sudo chmod 750 "${KAMIWAZA_SECURITY_DIR}"
    sudo chown "${FERNET_KEY_OWNER}:${FERNET_KEY_GROUP}" "${KAMIWAZA_SECURITY_DIR}"
    # Get fernet key
    ssh -i /etc/kamiwaza/ssl/cluster.key -o StrictHostKeyChecking=no ${KAMIWAZA_HEAD_IP} \
        "sudo cat ${KAMIWAZA_SECURITY_DIR}/fernet.key" | \
        sudo tee "${FERNET_KEY_PATH}" > /dev/null
    sudo chmod 600 "${FERNET_KEY_PATH}"
    kw_ensure_fernet_permissions "${FERNET_KEY_PATH}"
    kw_verify_fernet_key_readable "${FERNET_KEY_OWNER}" "${FERNET_KEY_PATH}"
else
    "${KAMIWAZA_ROOT}/.venv/bin/python3" util/generate_jwt_keys.py
    
    # Generate fernet key for head nodes and community edition
    if [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
        # Community edition: put fernet key in KAMIWAZA_ROOT/runtime
        mkdir -p ${KAMIWAZA_ROOT}/runtime
        if [[ ! -f ${KAMIWAZA_ROOT}/runtime/fernet.key ]]; then
            "${KAMIWAZA_ROOT}/.venv/bin/python3" -c "from kamiwaza.lib.util import generate_fernet_key; print(generate_fernet_key())" > \
                ${KAMIWAZA_ROOT}/runtime/fernet.key
            chmod 600 ${KAMIWAZA_ROOT}/runtime/fernet.key
        fi
    else
        # Enterprise edition: put fernet key in ${KAMIWAZA_SECURITY_DIR}
        sudo mkdir -p "${KAMIWAZA_SECURITY_DIR}"
        sudo chmod 750 "${KAMIWAZA_SECURITY_DIR}"
        sudo chown "${FERNET_KEY_OWNER}:${FERNET_KEY_GROUP}" "${KAMIWAZA_SECURITY_DIR}"
        if [[ ! -f "${FERNET_KEY_PATH}" ]]; then
            "${KAMIWAZA_ROOT}/.venv/bin/python3" -c "from kamiwaza.lib.util import generate_fernet_key; print(generate_fernet_key())" | \
                sudo tee "${FERNET_KEY_PATH}" > /dev/null
            sudo chmod 600 "${FERNET_KEY_PATH}"
        fi
        kw_ensure_fernet_permissions "${FERNET_KEY_PATH}"
        kw_verify_fernet_key_readable "${FERNET_KEY_OWNER}" "${FERNET_KEY_PATH}"
    fi
fi

# flag that we should run a non-disruptive db upgrade
if [[ "${KAMIWAZA_SWARM_HEAD:-}" == "true" ]]; then
    touch ${KAMIWAZA_ROOT}/.kamiwaza-db-reset
    echo "Set flag for db upgrade"
fi

# Get primary group of current user
PRIMARY_GROUP=$(id -gn)

# For community edition, logs go in KAMIWAZA_ROOT/logs
if [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
    LOG_DIR="/tmp"
else
    LOG_DIR="/opt/kamiwaza/logs"
fi

# Only set up kamiwazad service for enterprise edition
if [[ "${KAMIWAZA_COMMUNITY:-}" != "true" ]]; then
    # Install and start kamiwazad service
    sudo tee /etc/systemd/system/kamiwazad.service << EOF
[Unit]
Description=Kamiwaza Service Manager
After=network.target docker.service
Requires=docker.service

[Service]
Type=forking
PIDFile=/tmp/kamiwazad.pid
ExecStart=${KAMIWAZA_ROOT}/startup/kamiwazad.sh start
ExecStop=${KAMIWAZA_ROOT}/startup/kamiwazad.sh stop
Restart=always
RestartSec=30
StartLimitInterval=300
StartLimitBurst=5
User=${USER}
Group=${PRIMARY_GROUP}
WorkingDirectory=${KAMIWAZA_ROOT}
Environment=KAMIWAZA_ROOT=${KAMIWAZA_ROOT}
Environment=NVM_DIR=${NVM_BASE}
Environment=KAMIWAZA_COMMUNITY=${KAMIWAZA_COMMUNITY:-}
StandardOutput=append:${LOG_DIR}/kamiwazad.log
StandardError=append:${LOG_DIR}/kamiwazad.log

[Install]
WantedBy=multi-user.target
EOF

    sudo systemctl daemon-reload
    sudo systemctl enable kamiwazad
    sudo systemctl start kamiwazad

    chmod a+x ${KAMIWAZA_ROOT}/startup/kamiwazad.sh || true
fi

unset KAMIWAZA_INSTALL

# Create logs directory if it doesn't exist
if [[ ! -d "${LOG_DIR}" ]]; then
    # Try to create with sudo first in case we don't have permissions
    if ! sudo mkdir -p "${LOG_DIR}" 2>/dev/null; then
        # Fall back to regular mkdir if sudo fails
        mkdir -p "${LOG_DIR}" || {
            echo "ERROR: Failed to create logs directory at ${LOG_DIR}" 
            exit 1
        }
    fi
    
    # Set permissions to allow writing logs
    sudo chown -R "${USER}:${PRIMARY_GROUP}" "${LOG_DIR}" || {
        echo "WARNING: Failed to set permissions on ${LOG_DIR}"
        # Don't exit since the directory exists and may still be writable
    }
fi

# Mark as configured
touch ~/.kamiwaza-installed
