#!/bin/bash


set -euo pipefail

# Find and kill any running support sessions
echo "Looking for active support sessions..."

# Find all tmux sessions that match the hostname pattern
SESSIONS=$(tmux list-sessions 2>/dev/null | grep "${HOSTNAME}_[0-9]\+" | cut -d: -f1 || true)

if [ -z "$SESSIONS" ]; then
    echo "No active support sessions found."
    exit 0
fi

for SESSION in $SESSIONS; do
    echo "Found support session: $SESSION"
    
    # Get PIDs from tmux environment variables
    PIDS=$(tmux show-environment -t "$SESSION" | grep -E "SSH_PID|MONITOR_PID" | cut -d= -f2 || true)
    
    # Kill the PIDs gracefully first, then forcefully
    for PID in $PIDS; do
        if kill -0 "$PID" 2>/dev/null; then
            echo "Sending SIGTERM to process $PID..."
            kill "$PID" 2>/dev/null || true
            sleep 1
            if kill -0 "$PID" 2>/dev/null; then
                echo "Sending SIGKILL to process $PID..."
                kill -9 "$PID" 2>/dev/null || true
            fi
        fi
    done
    
    # Kill the tmux session
    echo "Terminating tmux session $SESSION..."
    tmux kill-session -t "$SESSION" 2>/dev/null || true
done

# Clean up orphaned processes after tmux sessions are gone
echo "Checking for orphaned socat processes..."
SOCAT_PIDS=$(ps aux | grep "socat TCP-LISTEN:" | grep -v grep | awk '{print $2}' || true)
if [ -n "$SOCAT_PIDS" ]; then
    echo "Found orphaned socat processes, terminating..."
    for PID in $SOCAT_PIDS; do
        kill "$PID" 2>/dev/null || true
        sleep 1
        kill -9 "$PID" 2>/dev/null || true
    done
fi

echo "Checking for orphaned SSH tunnels..."
SSH_PIDS=$(ps aux | grep "ssh.*-R.*sshuser@sra.kamiwaza.ai" | grep -v grep | awk '{print $2}' || true)
if [ -n "$SSH_PIDS" ]; then
    echo "Found orphaned SSH tunnels, terminating..."
    for PID in $SSH_PIDS; do
        kill "$PID" 2>/dev/null || true
        sleep 1
        kill -9 "$PID" 2>/dev/null || true
    done
fi

echo "Support session cleanup complete."


