#!/bin/bash

# Add preserve-etcd flag check
PRESERVE_ETCD=0
for arg in "$@"; do
    if [[ "$arg" == "--preserve-etcd" ]]; then
        PRESERVE_ETCD=1
        # Remove this arg from $@ so it doesn't interfere with component_arg
        shift
    fi
done

# Split EXCLUDE_CONTAINERS into an array if it is comma separated
EXCLUDE_CONTAINERS=${KAMIWAZA_EXCLUDE_CONTAINERS:-qdrant,nonexistant}
excluded_containers=(${EXCLUDE_CONTAINERS//,/ })

# Check for KAMIWAZA_ENV, set to "default" if not set
KAMIWAZA_ENV=${KAMIWAZA_ENV:-default}

# Get the component argument if provided
component_arg=$1

# Platform detection
if [[ "$(uname)" == "Darwin" ]]; then
    platform='osx'
else
    platform='linux'
fi

# Architecture detection
arch_raw=$(uname -m)
case "$arch_raw" in
    "x86_64")
        arch='amd64'
        ;;
    "aarch64"|"arm64")
        arch='arm64'
        ;;
    *)
        echo "Unsupported architecture: $arch_raw"
        exit 1
        ;;
esac

# GPU detection using perform_gpu_detection from common.sh
source "${KAMIWAZA_ROOT}/common.sh"
if [ "$(perform_gpu_detection)" = "true" ]; then
    cpugpu='gpu'
else
    cpugpu='cpu'
fi

# Main loop to process components
for component_path in $(find kamiwaza/deployment -mindepth 1 -maxdepth 1 -type d | grep -v '/envs$'); do
    component=$(basename $component_path)
    
    # Component arg filtering
    if [[ -n "$component_arg" && "$component" != *"$component_arg"* ]]; then
        continue
    fi

    # Exclusion check
    exclude=false
    for excluded in "${excluded_containers[@]}"; do
        if [[ "$component" == *"$excluded"* ]]; then
            exclude=true
            break
        fi
    done
    if [[ "$exclude" == true ]]; then
        echo "Skipping $component because it is excluded"
        continue
    fi

    # Architecture folder determination
    if [[ "$cpugpu" == "gpu" && -d "kamiwaza/deployment/envs/${KAMIWAZA_ENV}/${component}/${arch}-gpu" ]]; then
        arch_folder="${arch}-gpu"
    elif [[ -d "kamiwaza/deployment/envs/${KAMIWAZA_ENV}/${component}/${arch}" ]]; then
        arch_folder="${arch}"
    elif [[ "$cpugpu" == "cpu" && -d "kamiwaza/deployment/envs/${KAMIWAZA_ENV}/${component}/${arch}-cpu" ]]; then
        arch_folder="${arch}-cpu"
    else
        echo "No suitable architecture folder found for component: ${component}"
        continue
    fi

    # Navigate to component directory
    cd "kamiwaza/deployment/envs/${KAMIWAZA_ENV}/${component}/${arch_folder}"

    # Special handling for datahub - uses quickstart profile
    if [[ "$component" == *"datahub"* ]]; then
        echo "Stopping datahub stack"
        docker compose -f "docker-compose.yml" -p "${KAMIWAZA_ENV}-${component}" --profile quickstart down
    # Special handling for etcd
    elif [[ "$component" == *"etcd"* && "$PRESERVE_ETCD" -eq 0 ]]; then
        echo "Stopping etcd with volume removal"
        docker compose -f "docker-compose.yml" -p "${KAMIWAZA_ENV}-${component}" down
        # Actually remove the data directory
        rm -rf "${DOCKER_VOLUME_DIRECTORY:-.}/volumes/etcd"/*.etcd
    elif [[ "$component" == *"traefik"* ]]; then
        echo "Removing Traefik stack"
        docker compose -f "docker-compose.yml" -p "${KAMIWAZA_ENV}-${component}" down --remove-orphans
    # Special handling for Loki - always use down with volumes when not enabled
    elif [[ "$component" == *"loki"* && "${KAMIWAZA_LOKI_ENABLED:-false}" != "true" ]]; then
        echo "Removing Loki stack (not enabled)"
        docker compose -f "docker-compose.yml" -p "${KAMIWAZA_ENV}-${component}" down -v
    # Special handling for OTEL collector - handle Jaeger and OTEL separately based on flags
    elif [[ "$component" == *"otel"* ]]; then
        # Check both Jaeger and OTEL enabled flags
        jaeger_enabled="${KAMIWAZA_JAEGER_ENABLED:-false}"
        otel_enabled="${KAMIWAZA_OTEL_ENABLED:-false}"

        if [[ "$jaeger_enabled" != "true" && "$otel_enabled" != "true" ]]; then
            # Both disabled - remove everything with volumes
            echo "Removing OTEL collector stack (both Jaeger and OTEL disabled)"
            docker compose -f "docker-compose.yml" -p "${KAMIWAZA_ENV}-${component}" down -v
        elif [[ "$jaeger_enabled" != "true" ]]; then
            # Only Jaeger disabled - stop and remove Jaeger service only
            echo "Removing Jaeger from OTEL stack (Jaeger disabled)"
            docker compose -f "docker-compose.yml" -p "${KAMIWAZA_ENV}-${component}" rm -sf jaeger
        elif [[ "$otel_enabled" != "true" ]]; then
            # Only OTEL disabled - stop and remove OTEL collector only
            echo "Removing OTEL collector (OTEL disabled, keeping Jaeger)"
            docker compose -f "docker-compose.yml" -p "${KAMIWAZA_ENV}-${component}" rm -sf otel-collector
        else
            # Both enabled
            echo "Stopping OTEL collector stack"
            docker compose -f "docker-compose.yml" -p "${KAMIWAZA_ENV}-${component}" down --remove-orphans
        fi
    else
        echo "Stopping ${component}"
        docker compose -f "docker-compose.yml" -p "${KAMIWAZA_ENV}-${component}" down
    fi

    # Return to original directory
    cd - > /dev/null
done

# Stop frontend
echo "Stopping frontend..."
if [ -f "${KAMIWAZA_ROOT}/frontend/stop-frontend.sh" ]; then
    bash "${KAMIWAZA_ROOT}/frontend/stop-frontend.sh" || echo "Frontend stop script encountered an error (non-critical)"
else
    echo "Frontend stop script not found, trying manual shutdown..."
    if docker ps --filter "name=kamiwaza-frontend" --format "{{.Names}}" | grep -q "kamiwaza-frontend"; then
        cd "${KAMIWAZA_ROOT}/frontend" 2>/dev/null || true
        docker compose down 2>/dev/null || docker-compose down 2>/dev/null || true
        cd - > /dev/null 2>&1 || true
    fi
fi

echo "All containers have been stopped."
exit 0
