#!/bin/bash
set -euo pipefail

# from commit: a4b56d4d32557dc0db330698b311019d23260cee

# Helper functions
info() {
    echo "[INFO] $(date): $1"
}

# Configuration
BASTION_HOST="sra.kamiwaza.ai"
CUSTOMER_PORT=""
LOCAL_PORT="2222"  # Default port for local sshd
SUPPORT_PORT=""    # Support port will be assigned by the bastion
SSH_PRIVATE_KEY="$HOME/.ssh/reverse_tunnel_key"
MAX_RETRIES=5
RETRY_INTERVAL=10
LOG_FILE="$HOME/sra_tunnel.log"
MAX_LOG_SIZE=1048576  # 1MB in bytes
SSHD_DIR="$HOME/.sshd" # Define sshd dir
SSHD_CONFIG="$SSHD_DIR/sshd_config"
SSHD_ED25519_KEY="$SSHD_DIR/ssh_host_ed25519_key" # Define key path
SSHD_RSA_KEY="$SSHD_DIR/ssh_host_rsa_key" # Define key path
SSHD_PID_FILE="$SSHD_DIR/sshd.pid"
AUTOSSH_PIDFILE="$HOME/.ssh/autossh.pid"
DEBUG_MODE=false # Default to no debugging
DEBUG_LOG_FILE="$HOME/sra_debug.log" # Debug log file

RUN_FOREGROUND=false # Default to background/logging

# Parse command line arguments first
_args=("$@") # Store original arguments
while [[ $# -gt 0 ]]; do
    key="$1"
    case $key in
        -p|--port)
        if [[ -z "${2:-}" ]]; then echo "Error: Missing value for $1"; exit 1; fi
        CUSTOMER_PORT="$2"
        shift # past argument
        shift # past value
        ;;
        -l|--local-port)
        if [[ -z "${2:-}" ]]; then echo "Error: Missing value for $1"; exit 1; fi
        LOCAL_PORT="$2"
        shift # past argument
        shift # past value
        ;;
        -s|--support-port)
        if [[ -z "${2:-}" ]]; then echo "Error: Missing value for $1"; exit 1; fi
        SUPPORT_PORT="$2"
        shift # past argument
        shift # past value
        ;;
        --foreground)
        RUN_FOREGROUND=true
        shift # past argument
        ;;
        --debug)
        DEBUG_MODE=true
        echo "Debug mode enabled."
        shift # past argument
        ;;
        *)
        # Unknown option, pass it along (or handle error)
        shift
        ;;
    esac
done

# If CUSTOMER_PORT not set from command line, try environment variables
if [ -z "${CUSTOMER_PORT:-}" ]; then
    # First try CUSTOMER_PORT environment variable
    if [ ! -z "${CUSTOMER_PORT:-}" ]; then
        echo "Using port from CUSTOMER_PORT environment variable: $CUSTOMER_PORT"
    # Then try KAMIWAZA_CUSTOMER_PORT environment variable
    elif [ ! -z "${KAMIWAZA_CUSTOMER_PORT:-}" ]; then
        CUSTOMER_PORT="${KAMIWAZA_CUSTOMER_PORT}"
        echo "Using port from KAMIWAZA_CUSTOMER_PORT environment variable: $CUSTOMER_PORT"
    fi
fi

# Check if port is set after checking all sources
if [ -z "${CUSTOMER_PORT:-}" ]; then
    echo "Error: No customer port specified."
    echo ""
    echo "Usage: $0 -p <port> [-l <local-port>] [-s <support-port>] [--foreground] [--debug]"
    echo ""
    echo "  -p, --port <port>       : Bastion port for the customer (required)"
    echo "  -l, --local-port <port> : Local port to forward (default: 2222)"
    echo "  -s, --support-port <port>: Remote port on bastion (default: same as customer port)"
    echo "  --foreground            : Run in foreground and print logs to terminal instead of $LOG_FILE"
    echo "  --debug                 : Enable debug mode for more detailed logging"
    echo ""
    echo "Port can also be set via CUSTOMER_PORT or KAMIWAZA_CUSTOMER_PORT environment variables."
    exit 1
fi

# If support port not specified, use same as customer port 
if [ -z "${SUPPORT_PORT:-}" ]; then
    SUPPORT_PORT="${CUSTOMER_PORT}"
    echo "Using customer port as support port: $SUPPORT_PORT"
fi

# Function to clean up and exit
cleanup() {
    echo "Cleaning up and exiting..."
    # Kill the autossh process if its PID file exists
    if [ -f "$AUTOSSH_PIDFILE" ]; then
        local pid_to_kill=$(cat "$AUTOSSH_PIDFILE")
        if kill -0 $pid_to_kill 2>/dev/null; then
            echo "Stopping autossh process (PID: $pid_to_kill)..."
            kill $pid_to_kill
        fi
        rm -f "$AUTOSSH_PIDFILE"
    fi
    # Stop user-level sshd if its PID file exists
    if [ -f "$SSHD_PID_FILE" ]; then
        local sshd_pid_to_kill=$(cat "$SSHD_PID_FILE")
        if kill -0 $sshd_pid_to_kill 2>/dev/null; then
             echo "Stopping user-level sshd (PID: $sshd_pid_to_kill)..."
             kill -15 $sshd_pid_to_kill 2>/dev/null || true
        fi
    fi
    exit
}

# Set up trap for clean exit
trap cleanup INT TERM

# Function to rotate log file if it exceeds the maximum size
rotate_log_file() {
    if [ -f "$LOG_FILE" ] && [ $(stat -c%s "$LOG_FILE" 2>/dev/null || stat -f%z "$LOG_FILE") -ge $MAX_LOG_SIZE ]; then
        mv "$LOG_FILE" "$LOG_FILE.bak"
        echo "Log file rotated" > "$LOG_FILE"
    fi
}

# Check for required tools
for cmd in ssh autossh; do
    if ! command -v $cmd &> /dev/null; then
        echo "Error: Required command '$cmd' not found. Please install it and try again."
        exit 1
    fi
done

# Check if the SSH private key exists
if [ ! -f "$SSH_PRIVATE_KEY" ]; then
    echo "Error: Reverse tunnel key not found at $SSH_PRIVATE_KEY"
    echo "Please ensure you have placed the provided 'reverse_tunnel_key' in your ~/.ssh/ directory."
    exit 1
fi

# Install the forward tunnel key in authorized_keys if it exists in the deployment package
FORWARD_TUNNEL_KEY_PUB="$HOME/.ssh/forward_tunnel_key.pub"
if [ -f "$FORWARD_TUNNEL_KEY_PUB" ]; then
    echo "Installing Kamiwaza engineer access key to ~/.ssh/authorized_keys..."
    
    # Create .ssh directory if it doesn't exist
    mkdir -p "$HOME/.ssh"
    chmod 700 "$HOME/.ssh"
    
    # Create authorized_keys file if it doesn't exist
    touch "$HOME/.ssh/authorized_keys"
    chmod 600 "$HOME/.ssh/authorized_keys"
    
    # Check if the key is already installed
    KEY_CONTENT=$(cat "$FORWARD_TUNNEL_KEY_PUB")
    if grep -q "$KEY_CONTENT" "$HOME/.ssh/authorized_keys"; then
        echo "Kamiwaza engineer access key is already installed."
    else
        # Add the key with proper comment
        echo "# Kamiwaza engineer access key - Added by connect-support_v2.sh" >> "$HOME/.ssh/authorized_keys"
        cat "$FORWARD_TUNNEL_KEY_PUB" >> "$HOME/.ssh/authorized_keys"
        echo "Kamiwaza engineer access key has been installed successfully."
    fi
else
    echo "Warning: Kamiwaza engineer access key not found at $FORWARD_TUNNEL_KEY_PUB"
    echo "Engineers may not be able to connect to your system until this key is installed."
fi


# -- Set up logging redirection if not running in foreground --
if [ "$RUN_FOREGROUND" = false ]; then
    echo "Starting Kamiwaza SRA connection. Output will be logged to $LOG_FILE"
    # Redirect stdout and stderr to the log file for the rest of the script
    exec >>"$LOG_FILE" 2>&1
    echo "--- $(date): SRA Connection Script Started (PID: $$) ---"
fi

# Ensure sshd directory exists
if [ ! -d "$SSHD_DIR" ]; then
    echo "Creating SSH daemon directory ($SSHD_DIR)..."
    mkdir -p "$SSHD_DIR"
    chmod 700 "$SSHD_DIR"
else
    echo "SSH daemon directory ($SSHD_DIR) already exists."
fi

# Ensure sshd_config exists
if [ ! -f "$SSHD_CONFIG" ]; then
    echo "Creating default SSH daemon configuration ($SSHD_CONFIG)..."
    cat > "$SSHD_CONFIG" << EOF
Port 2222
ListenAddress 127.0.0.1
HostKey $SSHD_ED25519_KEY
HostKey $SSHD_RSA_KEY
AuthorizedKeysFile $HOME/.ssh/authorized_keys
PidFile $SSHD_PID_FILE
PermitRootLogin no
PasswordAuthentication no
PubkeyAuthentication yes
ChallengeResponseAuthentication no
UsePAM no
PermitEmptyPasswords no
AllowTcpForwarding yes
AllowAgentForwarding no
X11Forwarding no
#LogLevel DEBUG3
EOF
else
    echo "SSH daemon configuration ($SSHD_CONFIG) already exists."
fi

# Generate host keys if they don't exist
if [ ! -f "$SSHD_ED25519_KEY" ]; then
    echo "Generating Ed25519 host key ($SSHD_ED25519_KEY)..."
    ssh-keygen -t ed25519 -f "$SSHD_ED25519_KEY" -N ''
else
    echo "Ed25519 host key ($SSHD_ED25519_KEY) already exists."
fi

if [ ! -f "$SSHD_RSA_KEY" ]; then
    echo "Generating RSA host key ($SSHD_RSA_KEY)..."
    ssh-keygen -t rsa -b 4096 -f "$SSHD_RSA_KEY" -N ''
else
    echo "RSA host key ($SSHD_RSA_KEY) already exists."
fi

# Function to check if sshd is running
check_sshd_running() {
    if [ -f "$SSHD_PID_FILE" ]; then
        if kill -0 $(cat "$SSHD_PID_FILE") 2>/dev/null; then
            return 0 # Running
        else # Stale PID file
             echo "Warning: Found stale sshd PID file ($SSHD_PID_FILE). Removing it."
             rm -f "$SSHD_PID_FILE"
             return 1 # Not running
        fi
    fi
    return 1 # Not running
}

# Start the user-level SSH daemon if not already running
if ! check_sshd_running; then
    echo "Starting user-level SSH daemon..."
    /usr/sbin/sshd -f "$SSHD_CONFIG"
    sleep 2 # Give sshd time to start and create PID file
    
    if ! check_sshd_running; then
        echo "Error: Failed to start SSH daemon. Check your configuration ($SSHD_CONFIG) and logs."
        exit 1
    fi
    echo "SSH daemon started successfully (PID: $(cat $SSHD_PID_FILE))."
else
    echo "User-level SSH daemon is already running (PID: $(cat $SSHD_PID_FILE))."
fi

# Check if known hosts file exists and set verification accordingly
SRA_KNOWN_HOSTS="$HOME/.ssh/sra_known_hosts"
if [ -f "$SRA_KNOWN_HOSTS" ]; then
    echo "Found existing known hosts file: $SRA_KNOWN_HOSTS"
    STRICT_HOST_KEY_VERIFICATION=1
else
    echo "No existing known hosts file found. Will create on first connection."
    STRICT_HOST_KEY_VERIFICATION=0
fi

# Function to establish the reverse tunnel using autossh
establish_reverse_tunnel() {
    info "Establishing reverse tunnel"

    # Clear any existing HOST_KEY_VERIFICATION variable
    unset HOST_KEY_VERIFICATION
    
    # Configure the autossh environment
    export AUTOSSH_GATETIME=0  # allows immediate respawn attempts when client dies
    export AUTOSSH_POLL=60     # check every 60 seconds
    export AUTOSSH_MAXSTART=-1 # unlimited restarts (0 actually means no restarts allowed)
    export AUTOSSH_LOGLEVEL=7  # verbose logging level
    
    # Set strict host key checking if enabled
    if [ "$STRICT_HOST_KEY_VERIFICATION" = "1" ]; then
        STRICT_CHECKING="yes"
    else
        STRICT_CHECKING="no"
    fi

    # Determine autossh flags based on foreground/background mode
    local AUTOSSH_EXTRA_FLAGS=""
    local SSH_VERBOSITY=""
    
    if [ "$RUN_FOREGROUND" = false ]; then
        AUTOSSH_EXTRA_FLAGS="-f" # Run in background
        # AUTOSSH_PIDFILE env var tells autossh where to write its PID
        export AUTOSSH_PIDFILE
    else
        # In foreground mode, use verbose output
        AUTOSSH_EXTRA_FLAGS="-v"
        unset AUTOSSH_PIDFILE # Ensure not set from env
    fi
    
    # Add SSH verbosity if debug mode is enabled
    if [ "$DEBUG_MODE" = true ]; then
        SSH_VERBOSITY="-vvv" # Triple verbose for maximum SSH debugging
    fi
    
    rotate_log_file

    # Check if autossh is already running for this config
    if [ -f "$AUTOSSH_PIDFILE" ]; then
        if kill -0 $(cat "$AUTOSSH_PIDFILE") 2>/dev/null; then
            echo "Reverse tunnel (autossh) is already running (PID: $(cat $AUTOSSH_PIDFILE))."
            # If running in foreground, let the user know it's already running there
            if [ "$RUN_FOREGROUND" = true ]; then
                echo "Another instance seems to be running. Stop it first (kill $(cat $AUTOSSH_PIDFILE)) or run without --foreground."
            fi
            exit 0
        else
            echo "Warning: Found stale autossh PID file ($AUTOSSH_PIDFILE). Removing it."
            rm -f "$AUTOSSH_PIDFILE"
        fi
    fi

    echo "Establishing persistent reverse tunnel to $BASTION_HOST:$CUSTOMER_PORT..."
    export REMOTEUSER="$USER"
    
    # Set up debug environment variables if debug mode is enabled
    if [ "$DEBUG_MODE" = true ]; then
        export AUTOSSH_LOGFILE="$DEBUG_LOG_FILE"
        export AUTOSSH_DEBUG=1
        echo "Debug logging enabled. Debug log will be written to $DEBUG_LOG_FILE"
    fi
    
    # Use the already defined SRA_KNOWN_HOSTS variable
    if [ -f "$SRA_KNOWN_HOSTS" ]; then
        echo "Using existing known hosts file: $SRA_KNOWN_HOSTS"
    else
        echo "No existing known hosts file. Will create on first connection."
    fi

    # Start autossh with improved reliability settings
    autossh -M 0 $AUTOSSH_EXTRA_FLAGS \
        -o "SendEnv REMOTEUSER" \
        -o "ConnectTimeout 60" \
        -o "ServerAliveInterval 15" \
        -o "ServerAliveCountMax 3" \
        -o "StrictHostKeyChecking=$STRICT_CHECKING" \
        -o "UserKnownHostsFile=$SRA_KNOWN_HOSTS" \
        -o "ExitOnForwardFailure=yes" \
        -o "TCPKeepAlive=yes" \
        -R 127.0.0.1:2222:127.0.0.1:2222 \
        -R 127.0.0.1:8443:127.0.0.1:443 -R 127.0.0.1:8080:127.0.0.1:8080 \
        -R 127.0.0.1:8265:127.0.0.1:8265 -R 127.0.0.1:61100:127.0.0.1:61100 \
        -R 127.0.0.1:61101:127.0.0.1:61101 -R 127.0.0.1:61200:127.0.0.1:61200 \
        -p "$CUSTOMER_PORT" \
        -i "$SSH_PRIVATE_KEY" \
        $SSH_VERBOSITY \
        "kzcustomer@$BASTION_HOST"

    # Verify that autossh started correctly (only relevant for background mode)
    if [ "$RUN_FOREGROUND" = false ]; then
        sleep 5 # Give autossh time to start and write PID
        if [ ! -f "$AUTOSSH_PIDFILE" ]; then
             echo "Error: Failed to establish reverse tunnel. autossh PID file was not created."
             echo "Check the log file: $LOG_FILE"
             exit 1
        fi

        local AUTOSSH_PID=$(cat "$AUTOSSH_PIDFILE")
        if ! kill -0 $AUTOSSH_PID 2>/dev/null; then
            echo "Error: Failed to establish reverse tunnel. autossh process $AUTOSSH_PID is not running."
            echo "Check the log file: $LOG_FILE"
            rm -f "$AUTOSSH_PIDFILE" # Clean up bad PID file
            exit 1
        fi
        echo "Reverse tunnel established successfully in background."
        echo "autossh process PID: $AUTOSSH_PID"
        echo "Log file: $LOG_FILE"
    else
         echo "Reverse tunnel process started in foreground."
         # In foreground mode, autossh command above will block until exit/killed
         # The trap will handle cleanup.
         echo "--- $(date): SRA Connection Script Exiting (Foreground Mode) ---"
    fi
}

# Establish the initial reverse tunnel
establish_reverse_tunnel

# If running in background, print final messages and exit
if [ "$RUN_FOREGROUND" = false ]; then
    echo ""
    echo "Support access enabled."
    echo "The reverse tunnel is running in the background via autossh (PID: $(cat $AUTOSSH_PIDFILE))."
    echo "It will automatically reconnect if the connection drops."
    echo "Log file: $LOG_FILE"
    if [ "$DEBUG_MODE" = true ]; then
        echo "Debug log: $DEBUG_LOG_FILE"
    fi
    echo ""
    echo "To manually stop the tunnel, run:"
    echo "kill $(cat $AUTOSSH_PIDFILE)"
    echo ""
    echo "This script can be run directly or added to your crontab for automatic start on reboot:"
    # Use a more portable way to get the script path
    SCRIPT_PATH=$(cd "$(dirname "$0")" && pwd)/$(basename "$0")
    if [ "$DEBUG_MODE" = true ]; then
        echo "@reboot $SCRIPT_PATH -p $CUSTOMER_PORT -l $LOCAL_PORT -s $SUPPORT_PORT --debug >> $LOG_FILE.cron 2>&1"
    else
        echo "@reboot $SCRIPT_PATH -p $CUSTOMER_PORT -l $LOCAL_PORT -s $SUPPORT_PORT >> $LOG_FILE.cron 2>&1"
    fi

    # Script exits here, leaving autossh running in the background
    exit 0
fi

# If running in foreground, the script stays alive until autossh exits or is interrupted. 