#!/bin/bash

# Update only from support package, indicating the commit.
# connect-support.sh bcfb9c9df6f6e8b0466dfb2872b4a0d97fccb489

set -euo pipefail

# Configuration
BASTION_HOST="sra.kamiwaza.ai"
CUSTOMER_PORT=""
CUSTOMER_PRIVATE_KEY="$HOME/.ssh/kamiwaza_support_key"
CUSTOMER_KNOWN_HOSTS="$HOME/.ssh/known_hosts"
SESSION_NAME="${HOSTNAME}_$(date +%s)"
MAX_RETRIES=5
RETRY_INTERVAL=10
LOG_FILE="sra-connect.log"
MAX_LOG_SIZE=1048576  # 1MB in bytes

if [ -z "$CUSTOMER_PORT" ] && [ ! -z "$KAMIWAZA_CUSTOMER_PORT" ]; then
    CUSTOMER_PORT=$(printenv KAMIWAZA_CUSTOMER_PORT)
fi



# Function to clean up and exit
cleanup() {
    echo "Cleaning up and exiting..."
    [[ -n "${SSH_PID:-}" ]] && kill $SSH_PID 2>/dev/null
    tmux kill-session -t "$SESSION_NAME" 2>/dev/null
    exit
}

# Set up trap for clean exit
trap cleanup INT TERM

# Function to rotate log file if it exceeds the maximum size
rotate_log_file() {
    if [ -f "$LOG_FILE" ] && [ $(stat -c%s "$LOG_FILE") -ge $MAX_LOG_SIZE ]; then
        mv "$LOG_FILE" "$LOG_FILE.bak"
        echo "Log file rotated" > "$LOG_FILE"
    fi
}

# Parse command line arguments
while [[ $# -gt 0 ]]; do
    key="$1"
    case $key in
        -p|--port)
        CUSTOMER_PORT="$2"
        shift
        shift
        ;;
        *)
        echo "Unknown option: $1"
        exit 1
        ;;
    esac
done

# Check if required argument is provided
if [ -z "$CUSTOMER_PORT" ]; then
    echo "Usage: $0 -p <port>"
    exit 1
fi

# Check for required tools
for cmd in tmux ssh socat; do
    if ! command -v $cmd &> /dev/null; then
        echo "Error: Required command '$cmd' not found. Please install it and try again."
        exit 1
    fi
done

# Check if the customer private key exists
if [ ! -f "$CUSTOMER_PRIVATE_KEY" ]; then
    echo "Error: Kamiwaza support key not found at $CUSTOMER_PRIVATE_KEY"
    echo "Please ensure you have placed the provided 'kamiwaza_support_key' in your ~/.ssh/ directory."
    exit 1
fi

# Check if the known_hosts file exists
if [ ! -f "$CUSTOMER_KNOWN_HOSTS" ]; then
    echo "Error: Known hosts file not found at $CUSTOMER_KNOWN_HOSTS"
    echo "Please ensure you have appended the provided known_hosts entry to your ~/.ssh/known_hosts file."
    exit 1
fi

# Function to establish SSH connection
establish_ssh_connection() {
    rotate_log_file
    ssh -i "$CUSTOMER_PRIVATE_KEY" \
        -o UserKnownHostsFile="$CUSTOMER_KNOWN_HOSTS" \
        -o StrictHostKeyChecking=yes \
        -o ServerAliveInterval=60 \
        -o ServerAliveCountMax=3 \
        -o TCPKeepAlive=yes \
        -R "$CUSTOMER_PORT":localhost:"$CUSTOMER_PORT" \
        -R 8080:localhost:80 \
        -R 8443:localhost:443 \
        -R 8265:localhost:8265 \
        -R 61100:localhost:61100 \
        -R 61101:localhost:61101 \
        -R 61200:localhost:61200 \
        -p "$CUSTOMER_PORT" \
        "sshuser@$BASTION_HOST" \
        -N >> "$LOG_FILE" 2>&1 &
    SSH_PID=$!
}

monitor_ssh_connection() {
    local consecutive_failures=0

    while true; do
        if kill -0 $SSH_PID 2>/dev/null; then
            # Connection is alive, reset the consecutive failure count
            if [ $consecutive_failures -ne 0 ]; then
                echo "SSH connection is stable. Resetting failure count." >> "$LOG_FILE"
                consecutive_failures=0
            fi
        else
            echo "SSH connection lost. Waiting before attempting to reconnect..." >> "$LOG_FILE"
            consecutive_failures=$((consecutive_failures + 1))
            
            for ((i=1; i<=MAX_RETRIES; i++)); do
                # Wait before each attempt, including the first one
                sleep $RETRY_INTERVAL
                echo "Retry attempt $i of $MAX_RETRIES (Consecutive failures: $consecutive_failures)" >> "$LOG_FILE"
                establish_ssh_connection
                # Short wait to allow SSH to start up
                sleep 5
                if kill -0 $SSH_PID 2>/dev/null; then
                    echo "SSH connection re-established." >> "$LOG_FILE"
                    break
                fi
                if [ $i -eq $MAX_RETRIES ]; then
                    if [ $consecutive_failures -ge $MAX_RETRIES ]; then
                        echo "Failed to re-establish SSH connection after $MAX_RETRIES consecutive failure sets." >> "$LOG_FILE"
                        cleanup
                    else
                        echo "Max retries reached for this failure set. Will continue monitoring." >> "$LOG_FILE"
                        break
                    fi
                fi
            done
        fi
        # Regular check interval
        sleep 10
    done
}

# Create a new tmux session
tmux new-session -d -s "$SESSION_NAME" -x 200 -y 54 "bash"
tmux set-option -t "$SESSION_NAME" remain-on-exit on

# Capture the TTY of the first pane (bash)
TTY_PANE_0=$(tmux display -p -t "$SESSION_NAME:0.0" '#{pane_tty}')

# Use socat to handle the TTY connection
tmux split-window -v -t "$SESSION_NAME" "socat TCP-LISTEN:$CUSTOMER_PORT,reuseaddr,fork EXEC:'tmux attach-session -t $SESSION_NAME',pty,stderr,setsid,sigint,sane"

# Establish initial SSH connection
establish_ssh_connection

# Start monitoring SSH connection in the background
monitor_ssh_connection &
MONITOR_PID=$!

# Store PIDs in tmux environment for reliable cleanup
tmux set-environment -t "$SESSION_NAME" "SSH_PID" "$SSH_PID"
tmux set-environment -t "$SESSION_NAME" "MONITOR_PID" "$MONITOR_PID"

# Send output to the first tmux pane (bash)
echo "Support access successfully enabled." > "$TTY_PANE_0"
echo "Support session name: $SESSION_NAME" > "$TTY_PANE_0"
echo "SSH PID: $SSH_PID" > "$TTY_PANE_0"
echo "Monitor PID: $MONITOR_PID" > "$TTY_PANE_0"
echo "You are now viewing the support session." > "$TTY_PANE_0"
echo " - You can detach by pressing 'Ctrl-B d'" > "$TTY_PANE_0"
echo " - Re-attach to the session (the interactive shell): tmux attach-session -t $SESSION_NAME:0" > "$TTY_PANE_0"
echo " - To forcibly terminate, run: tmux kill-session -t $SESSION_NAME && kill $SSH_PID $MONITOR_PID" > "$TTY_PANE_0"
echo " - You may want to COPY THOSE COMMANDS as they will not be displayed again." > "$TTY_PANE_0"

sleep 1
tmux attach-session -t "$SESSION_NAME:0"
