#!/bin/bash
#!/usr/bin/env bash

# Shared Python helpers for Kamiwaza shell tooling.
# Shell scripts can source this file to access uv-backed runners.

kw_py_hash_file() {
    local target="$1"
    local project_dir="$2"
    local canonical="$target"

    if [[ -z "$target" ]]; then
        echo "unknown"
        return 1
    fi

    if command -v python3 >/dev/null 2>&1; then
        if ! canonical=$(python3 -c 'import os, sys; print(os.path.realpath(sys.argv[1]))' "$target" 2>/dev/null); then
            echo "unknown"
            return 1
        fi
    elif command -v realpath >/dev/null 2>&1; then
        if ! canonical=$(realpath "$target" 2>/dev/null); then
            echo "unknown"
            return 1
        fi
    elif command -v readlink >/dev/null 2>&1; then
        canonical=$(readlink -f "$target" 2>/dev/null || printf '%s' "$target")
    fi

    if [[ ! -f "$canonical" ]]; then
        echo "missing"
        return 0
    fi

    if [[ -n "$project_dir" ]]; then
        case "$canonical" in
            "$project_dir"/*) ;;
            *) echo "unknown"; return 1 ;;
        esac
    fi

    if command -v shasum >/dev/null 2>&1; then
        shasum -a 256 "$canonical" | awk '{print $1}'
        return 0
    fi

    if command -v sha256sum >/dev/null 2>&1; then
        sha256sum "$canonical" | awk '{print $1}'
        return 0
    fi

    echo "unknown"
    return 1
}

kw_py_pip_bootstrap_env() {
    local project_dir="$1"
    local host_python="${PYTHON3_BIN:-}"

    if [[ -z "$host_python" ]]; then
        if command -v python3 >/dev/null 2>&1; then
            host_python="$(command -v python3)"
        elif command -v python >/dev/null 2>&1; then
            host_python="$(command -v python)"
        fi
    fi

    if [[ -z "$host_python" ]]; then
        echo "[kw_py] no system python available for pip fallback bootstrap" >&2
        return 1
    fi

    if [[ ! -d "${project_dir}/.venv" ]]; then
        "$host_python" -m venv "${project_dir}/.venv"
    fi
    "$host_python" -m venv --upgrade-deps "${project_dir}/.venv"

    local venv_python="${project_dir}/.venv/bin/python"
    if [[ ! -x "$venv_python" ]]; then
        echo "[kw_py] unable to locate managed interpreter at ${venv_python}" >&2
        return 1
    fi

    "$venv_python" -m ensurepip --upgrade >/dev/null 2>&1 || true

    "$venv_python" -m pip install --upgrade pip setuptools wheel >&2

    local requirements_file="${project_dir}/requirements.txt"
    if [[ ! -f "$requirements_file" ]]; then
        echo "[kw_py] requirements.txt not found; cannot bootstrap dependencies" >&2
        return 1
    fi

    "$venv_python" -m pip install -r "$requirements_file" >&2
}

kw_py_verify_env_ready() {
    local project_dir="$1"
    local managed_python="${project_dir}/.venv/bin/python"

    if [[ ! -x "$managed_python" ]]; then
        return 1
    fi

    "$managed_python" - <<'PY' >/dev/null 2>&1
import importlib.util
import sys
required = ("pydantic", "fastapi")
missing = [m for m in required if importlib.util.find_spec(m) is None]
sys.exit(0 if not missing else 1)
PY
}

kw_py_bootstrap_uv_env() {
    local project_dir="$1"

    if [[ "${KW_PY_NO_SYNC:-}" != "1" ]]; then
        return 0
    fi

    local marker_dir="${project_dir}/.kw"
    local marker_file="${marker_dir}/kw_py.ready"
    local pip_mode_file="${marker_dir}/kw_py.force_pip"
    local managed_python="${project_dir}/.venv/bin/python"
    local needs_sync=0
    local current_py_hash current_lock_hash
    local KW_PY_MARKER_PY_HASH=""
    local KW_PY_MARKER_LOCK_HASH=""
    local pip_only="false"

    current_py_hash=$(kw_py_hash_file "${project_dir}/pyproject.toml" "$project_dir")
    current_lock_hash=$(kw_py_hash_file "${project_dir}/uv.lock" "$project_dir")
    if [[ -f "$pip_mode_file" ]]; then
        pip_only="true"
    fi

    if [[ ! -x "$managed_python" ]]; then
        needs_sync=1
    elif [[ ! -f "$marker_file" ]]; then
        needs_sync=1
    else
        # shellcheck disable=SC1090
        source "$marker_file" >/dev/null 2>&1 || true
        if [[ "$KW_PY_MARKER_PY_HASH" != "$current_py_hash" ]] || [[ "$KW_PY_MARKER_LOCK_HASH" != "$current_lock_hash" ]]; then
            needs_sync=1
        fi
    fi

    if [[ "$needs_sync" -ne 1 ]]; then
        if kw_py_verify_env_ready "$project_dir"; then
            return 0
        fi
        needs_sync=1
    fi

    mkdir -p "$marker_dir"
    echo "[kw_py] KW_PY_NO_SYNC=1 but dependencies are missing/stale; running uv_sync.sh once" >&2
    if [[ "$pip_only" != "true" ]]; then
        if [[ -x "${project_dir}/uv_sync.sh" ]]; then
            (
                cd "$project_dir"
                KW_PY_NO_SYNC=0 KW_UV_NO_DEFAULT=1 KW_UV_GROUPS='' ./uv_sync.sh >&2
            ) || echo "[kw_py] uv_sync.sh failed; attempting pip-based fallback install" >&2
        else
            uv --project "$project_dir" sync || echo "[kw_py] 'uv sync' failed; attempting pip-based fallback install" >&2
        fi
    else
        echo "[kw_py] uv bootstrap disabled (pip-only mode)" >&2
    fi

    if ! kw_py_verify_env_ready "$project_dir"; then
        echo "[kw_py] managed environment still missing core dependencies after uv sync; attempting pip fallback" >&2
        if ! kw_py_pip_bootstrap_env "$project_dir"; then
            echo "[kw_py] unable to bootstrap Python environment via uv or pip" >&2
            return 1
        fi
        touch "$pip_mode_file"
        if ! kw_py_verify_env_ready "$project_dir"; then
            echo "[kw_py] managed environment still missing core dependencies after bootstrap" >&2
            return 1
        fi
    fi

    local final_py_hash final_lock_hash
    final_py_hash=$(kw_py_hash_file "${project_dir}/pyproject.toml" "$project_dir")
    final_lock_hash=$(kw_py_hash_file "${project_dir}/uv.lock" "$project_dir")
    {
        echo "KW_PY_MARKER_PY_HASH=\"$final_py_hash\""
        echo "KW_PY_MARKER_LOCK_HASH=\"$final_lock_hash\""
        echo "KW_PY_MARKER_TIMESTAMP=\"$(date -u +%s)\""
    } >"$marker_file"

    if [[ "$pip_only" == "true" && ! -f "$pip_mode_file" ]]; then
        touch "$pip_mode_file"
    fi
}

kw_py() {
    # Run Python through uv within the Kamiwaza project.
    # Falls back to invoking python directly if uv is unavailable.
    local project_dir="${KAMIWAZA_ROOT:-${KW_PROJECT_ROOT:-}}"

    if [[ -z "$project_dir" ]]; then
        project_dir="$(pwd)"
    fi

    # Guard against foreign virtual environments that could interfere with uv
    if [[ -n "${VIRTUAL_ENV:-}" ]]; then
        local _our_venv="${project_dir}/.venv"
        local _is_foreign=true
        if [[ -d "${VIRTUAL_ENV}" ]] && [[ -d "$_our_venv" ]]; then
            local _resolved_active _resolved_ours
            _resolved_active="$(cd "${VIRTUAL_ENV}" && pwd)"
            _resolved_ours="$(cd "$_our_venv" && pwd)"
            if [[ "$_resolved_active" == "$_resolved_ours" ]]; then
                _is_foreign=false
            fi
        fi
        if [[ "$_is_foreign" == "true" ]]; then
            echo "[kw_py] Clearing foreign virtual environment: ${VIRTUAL_ENV}" >&2
            local _foreign_bin="${VIRTUAL_ENV}/bin"
            local _new_path="" _old_ifs="$IFS" _dir
            IFS=':'
            for _dir in $PATH; do
                if [[ "$_dir" != "$_foreign_bin" ]]; then
                    _new_path="${_new_path:+${_new_path}:}${_dir}"
                fi
            done
            IFS="$_old_ifs"
            PATH="$_new_path"
            unset VIRTUAL_ENV
        fi
    fi

    local runner="${KW_PY_RUNNER_PATH:-${project_dir}/scripts/kw_py}"
    if [[ -x "$runner" ]]; then
        "$runner" "$@"
        return $?
    fi

    if command -v uv >/dev/null 2>&1; then
        kw_py_bootstrap_uv_env "$project_dir"
        local uv_cmd=(uv --project "$project_dir" run)
        # Allow callers to opt out of sync if environment guaranteed ready.
        if [[ "${KW_PY_NO_SYNC:-}" == "1" ]]; then
            uv_cmd+=("--frozen")
        fi
        uv_cmd+=("python")
        uv_cmd+=("$@")
        "${uv_cmd[@]}"
    else
        echo "[kw_py] uv not found on PATH; falling back to system python" >&2
        if command -v python3 >/dev/null 2>&1; then
            python3 "$@"
        else
            python "$@"
        fi
    fi
}
