#!/bin/bash
set -e

if [[ "$(uname)" != "Linux" ]]; then
    echo "Kamiwaza cluster mode is only supported on Linux - will not prep cluster"
    exit 1
fi

# Determine the target app user we should provision for.
TARGET_USER="${KAMIWAZA_TARGET_USER:-}"
TARGET_GROUP="${KAMIWAZA_TARGET_GROUP:-$TARGET_USER}"

if [[ -z "$TARGET_USER" ]]; then
    TARGET_USER=$(who am i | awk '{print $1}')
    if [[ -z "$TARGET_USER" ]]; then
        TARGET_USER=${SUDO_USER:-$USER}
    fi
    TARGET_GROUP="$TARGET_USER"
fi

CURRENT_USER=$(whoami)
if [[ "$CURRENT_USER" != "root" && "$CURRENT_USER" != "$TARGET_USER" ]]; then
    echo "Error: This script must be run as either root or $TARGET_USER"
    echo "Current user: $CURRENT_USER"
    exit 1
fi

# Test if we can use sudo without prompting for a password
if ! sudo -n true 2>/dev/null; then
    echo "This script requires sudo privileges without a password prompt. Please ensure you have the necessary sudo access."
    exit 1
fi

# Parse arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --head)
            IS_HEAD=true
            shift
            ;;
        --worker)
            IS_HEAD=false
            shift
            ;;
        --head-ip=*)
            HEAD_IP="${1#*=}"
            shift
            ;;
        *)
            echo "Usage: $0 --head | --worker --head-ip=<ip>"
            exit 1
            ;;
    esac
done

# Validate arguments
if [ -z "$IS_HEAD" ]; then
    echo "Usage: $0 --head | --worker --head-ip=<ip>"
    exit 1
fi

if [[ "$IS_HEAD" != "true" && -z "$HEAD_IP" ]]; then
    echo "Worker nodes require --head-ip parameter"
    exit 1
fi

echo "Preparing system for manual cluster setup..."

# Create directories
sudo mkdir -p /etc/kamiwaza/ssl
sudo chown $TARGET_USER:$TARGET_GROUP /etc/kamiwaza
sudo chmod 750 /etc/kamiwaza

# Generate SSH key pair if on head node
if [[ "$IS_HEAD" == "true" ]]; then
    # Ensure ~/.ssh exists with right permissions on head
    mkdir -p ~/.ssh
    chmod 700 ~/.ssh

    if [[ ! -f /etc/kamiwaza/ssl/cluster.key ]]; then
        # Generate the key with .key suffix
        sudo ssh-keygen -t rsa -b 4096 -f /etc/kamiwaza/ssl/cluster.key -N "" -C "kamiwaza-cluster"
        sudo mv /etc/kamiwaza/ssl/cluster.key.pub /etc/kamiwaza/ssl/cluster.pub
    fi
        
    # Check authorized_keys regardless of whether we just generated the key
    if ! grep -qFf <(sudo cat /etc/kamiwaza/ssl/cluster.pub) ~/.ssh/authorized_keys 2>/dev/null; then
        sudo cat /etc/kamiwaza/ssl/cluster.pub >> ~/.ssh/authorized_keys
        chmod 600 ~/.ssh/authorized_keys
    fi
    
echo "========================================================="
    echo "Key distribution instructions for worker nodes:"
    echo ""
    echo "1. First create required directories and set permissions:"
    echo "   sudo mkdir -p /etc/kamiwaza/ssl"
echo "   sudo chmod 750 /etc/kamiwaza"
echo "   sudo chown $TARGET_USER:$TARGET_GROUP /etc/kamiwaza"
    echo ""
    echo "2. Copy cluster keys to worker node:"
    echo "   # Copy private key"
    echo "   sudo tee /etc/kamiwaza/ssl/cluster.key << 'KEYEOF' > /dev/null"
    sudo cat /etc/kamiwaza/ssl/cluster.key
    echo "KEYEOF"
    echo ""
    echo "   # Copy public key"
    echo "   sudo tee /etc/kamiwaza/ssl/cluster.pub << 'PUBEOF' > /dev/null"
    sudo cat /etc/kamiwaza/ssl/cluster.pub
    echo "PUBEOF"
    echo ""
    echo "3. Set key permissions:"
    echo "   sudo chmod 600 /etc/kamiwaza/ssl/cluster.key"
    echo "   sudo chmod 644 /etc/kamiwaza/ssl/cluster.pub"
    echo ""
    echo "4. Add public key to authorized_keys:"
    echo "   mkdir -p ~/.ssh"
    echo "   chmod 700 ~/.ssh"
    echo "   cat /etc/kamiwaza/ssl/cluster.pub >> ~/.ssh/authorized_keys"
    echo "   chmod 600 ~/.ssh/authorized_keys"
    echo "========================================================="
fi
# For worker nodes, ensure ~/.ssh exists with right permissions
if [[ "$IS_HEAD" != "true" ]]; then
    mkdir -p ~/.ssh
    chmod 700 ~/.ssh
fi

validate_keys() {
    mkdir -p ~/.ssh
    chmod 700 ~/.ssh
    if [[ "$IS_HEAD" == "true" ]]; then
        if [[ ! -f /etc/kamiwaza/ssl/cluster.key ]] || [[ ! -f /etc/kamiwaza/ssl/cluster.pub ]]; then
            echo "Error: Missing cluster keys on head node"
            exit 1
        fi
        if ! grep -q "kamiwaza-cluster" ~/.ssh/authorized_keys 2>/dev/null; then
            echo "Error: Public key not found in head node authorized_keys"
            exit 1
        fi
    else
        if [[ ! -f /etc/kamiwaza/ssl/cluster.key ]]; then
            echo "Error: Missing cluster.key on worker node"
            exit 1
        fi
        if ! grep -q "kamiwaza-cluster" ~/.ssh/authorized_keys 2>/dev/null; then
            echo "Error: Public key not found in worker authorized_keys"
            exit 1
        fi
    fi
}

# Set permissions on both files
sudo chown -R $TARGET_USER:$TARGET_GROUP /etc/kamiwaza
sudo chmod 600 /etc/kamiwaza/ssl/cluster.key
sudo chmod 644 /etc/kamiwaza/ssl/cluster.pub

# Configure SSH forwarding
sudo sed -i 's/#AllowAgentForwarding yes/AllowAgentForwarding yes/' /etc/ssh/sshd_config
sudo systemctl restart sshd

if [ -f /etc/kamiwaza/env.sh ]; then    
    sudo chmod 640 /etc/kamiwaza/env.sh
fi

# Validate key setup
validate_keys || { 
    echo "Key validation failed. Please check key distribution and permissions."
    exit 1
}

# If we get here, all went well
echo "Cluster prep completed successfully."
