#!/bin/bash

# Parse command line arguments
OFFLINE_MODE=false
while [[ $# -gt 0 ]]; do
    case $1 in
        --offline)
            OFFLINE_MODE=true
            shift
            ;;
        *)
            log_error "Unknown option: $1"
            exit 1
            ;;
    esac
done

# Export for potential future use
export OFFLINE_MODE

# Set verbose mode based on conditions
if [[ "${KAMIWAZA_DEBUG}" == "true" || -f kamiwaza-shibboleth ]]; then
    set -ex
else
    set -e
fi

# Determine script directory and key absolute paths
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
LLAMACPP_DIR="${SCRIPT_DIR}/llamacpp"
LLAMA_SRC_DIR="${LLAMACPP_DIR}/llama.cpp"
OFFLINE_TARBALL="${SCRIPT_DIR}/llama.cpp.tar.gz"

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
NC='\033[0m' # No Color
# Logging functions
log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"

}
log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}
log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}
log_header() {
    echo -e "${BLUE}=== $1 ===${NC}"
}
log_success() {
    echo -e "${PURPLE}[SUCCESS]${NC} $1"
}

# Determine llama.cpp commit: env var > file > default
LLAMA_COMMIT="${KAMIWAZA_LLAMA_COMMIT:-}"
if [[ -z "${LLAMA_COMMIT}" ]]; then
    if [[ -f "${SCRIPT_DIR}/llamacpp.commit" ]]; then
        LLAMA_COMMIT="$(<"${SCRIPT_DIR}/llamacpp.commit")"
    else
        LLAMA_COMMIT="da95bf2"
    fi
fi

# Check if commit is empty after all attempts
if [[ -z "${LLAMA_COMMIT}" ]]; then
    log_error "No commit specified via KAMIWAZA_LLAMA_COMMIT, llamacpp.commit file, or default"
    exit 1
fi

USING_PREPACKAGED=0

extract_tarball() {
    local tar_file="$1"
    local dest_dir="$2"

    mkdir -p "$dest_dir"

    if command -v tar >/dev/null 2>&1; then
        tar -xzf "$tar_file" -C "$dest_dir"
    else
        local py_bin
        if command -v python3 >/dev/null 2>&1; then
            py_bin="$(command -v python3)"
        elif command -v python >/dev/null 2>&1; then
            py_bin="$(command -v python)"
        else
            log_error "Neither tar nor python interpreter is available to extract $tar_file"
            return 1
        fi
        "${py_bin}" - "$tar_file" "$dest_dir" <<'PY'
import sys
from pathlib import Path
import tarfile

tar_path = Path(sys.argv[1])
dest_path = Path(sys.argv[2])
dest_path.mkdir(parents=True, exist_ok=True)

with tarfile.open(tar_path) as tf:
    tf.extractall(dest_path)
PY
    fi
}

try_prepackaged_bundle() {
    if [ "${KAMIWAZA_FORCE_LLAMA_BUILD:-0}" = "1" ]; then
        return 1
    fi

    if [ ! -f "$OFFLINE_TARBALL" ]; then
        return 1
    fi

    log_info "Found prepackaged llama.cpp bundle at $OFFLINE_TARBALL; attempting to use it"
    rm -rf "$LLAMACPP_DIR"
    mkdir -p "$LLAMACPP_DIR"

    if extract_tarball "$OFFLINE_TARBALL" "$LLAMACPP_DIR"; then
        USING_PREPACKAGED=1
        log_success "Prepackaged llama.cpp source bundle extracted"
        return 0
    else
        log_warn "Failed to extract prepackaged llama.cpp bundle; falling back to source build"
        rm -rf "$LLAMACPP_DIR"
        return 1
    fi
}
# Function to detect WSL environment
is_wsl() {
    if uname_out="$(uname -a 2>/dev/null)"; then
        if echo "$uname_out" | grep -q "microsoft-standard-WSL2"; then
            return 0  # true
        fi
    fi
    return 1  # false
}

# Function to check oneAPI installation
check_oneapi() {
    if [ -f "/opt/intel/oneapi/setvars.sh" ]; then
        log_info "Intel oneAPI found"
        return 0
    else
        log_warn "Intel oneAPI not found at /opt/intel/oneapi/setvars.sh"
        return 1
    fi
}
# Function to build llama.cpp with SYCL (Intel GPU)
build_llama_sycl() {
    local build_dir=$1
    cd "$build_dir"
    log_info "Building llama.cpp with SYCL support for Intel GPU..."
    # Source oneAPI environment
    # shellcheck source=/dev/null
    source /opt/intel/oneapi/setvars.sh
    # Clean and build with SYCL
    rm -rf build
    mkdir -p build && cd build
    # Configure with SYCL support
    cmake .. -DGGML_SYCL=ON -DCMAKE_C_COMPILER=icx -DCMAKE_CXX_COMPILER=icpx
    # Build with all available cores
    make -j"$(nproc)"
    cd ..
}

# Function to build llama.cpp (standard)
build_llama_standard() {

    local build_dir=$1
    cd "$build_dir"


    local python_bin="python3"
    if command -v python3.12 >/dev/null 2>&1; then
        python_bin="python3.12"
    elif command -v python3.11 >/dev/null 2>&1; then
        python_bin="python3.11"
    elif command -v python3.10 >/dev/null 2>&1; then
        python_bin="python3.10"
    elif command -v python >/dev/null 2>&1; then
        python_bin="python"
    fi
    if [ ! -d ".venv" ]; then
        log_info "Creating virtual environment for llama.cpp build dependencies"
        "$python_bin" -m venv .venv
    fi

    if command -v uv >/dev/null 2>&1; then
        log_info "Installing build requirements with uv into $(pwd)/.venv"
        uv pip install --python "$(pwd)/.venv/bin/python" -r requirements.txt
    else
        log_warn "uv not found; falling back to venv-local pip install"
        "$python_bin" -m pip install --pre -r requirements.txt
    fi
    log_info "Building llama.cpp with standard configuration..."
    if [ -d ".git" ] && [ "$USING_PREPACKAGED" = "0" ]; then
        git fetch origin || log_warn "git fetch failed; continuing with existing sources"
        git checkout "$LLAMA_COMMIT" || log_warn "git checkout $LLAMA_COMMIT failed; continuing"
        git branch -D kamiwaza || true
        git checkout -b kamiwaza || log_warn "git checkout -b kamiwaza failed; continuing"
    else
        log_info "Skipping git remote operations (offline bundle or missing metadata)"
    fi
    cmake -B build -DGGML_LTO=ON -DCMAKE_BUILD_TYPE=Release
    cmake --build build --config Release
    cd ..
}
# Main build function
build_llama() {
    local build_dir=$1
    local build_type=$2
    case "$build_type" in
        "sycl")
            build_llama_sycl "$build_dir"
            ;;
        "standard")
            build_llama_standard "$build_dir"
            ;;
        *)
            log_error "Unknown build type: $build_type"
            return 1
            ;;
    esac
}
# Add dependency checks
check_dependencies() {
	local missing=0
	if ! command -v cmake >/dev/null 2>&1; then
		log_error "cmake is not installed. Please install cmake."
		missing=1
	fi
	if ! command -v make >/dev/null 2>&1; then
		log_warn "make not found; some generators require it. Install build-essential if missing."
	fi
	if [ "$missing" -ne 0 ]; then
		return 1
	fi
	return 0
}
# Main execution
main() {
    log_header "Building llama.cpp for Kamiwaza"

    try_prepackaged_bundle || true
    # Ensure dependencies are available
    if ! check_dependencies; then
        log_error "Missing required dependencies. Install cmake (and make) then rerun."
        exit 1
    fi
    # Check if we're in WSL
    # Source common.sh for GPU detection
    source "${SCRIPT_DIR}/common.sh"
    
    # Determine build type based on environment and hardware
    gpu_result=$(perform_gpu_detection true)
    if is_wsl && [ "$gpu_result" = "true" ] && check_oneapi; then
        log_info "WSL with Intel GPU detected, using SYCL build"
        BUILD_TYPE="sycl"
    else
        log_info "Using standard build"
        BUILD_TYPE="standard"
    fi
    # Ensure working directory exists
    mkdir -p "$LLAMACPP_DIR"
    cd "$LLAMACPP_DIR"
    if [ "$USING_PREPACKAGED" = "1" ]; then
        if [ ! -d "$LLAMA_SRC_DIR" ]; then
            log_error "Offline llama.cpp bundle missing extracted sources"
            exit 1
        fi
        log_info "Using prepackaged llama.cpp sources (offline mode)"
    else
        if [ -d "$LLAMA_SRC_DIR" ]; then
            log_info "Existing llama.cpp found. Re-building from upstream..."
            mv "$LLAMA_SRC_DIR" "${LLAMA_SRC_DIR}_old"
            if git clone https://github.com/ggml-org/llama.cpp.git "$LLAMA_SRC_DIR"; then
                rm -rf "${LLAMA_SRC_DIR}_old"
            else
                log_warn "Clone failed. Restoring previous sources."
                rm -rf "$LLAMA_SRC_DIR"
                mv "${LLAMA_SRC_DIR}_old" "$LLAMA_SRC_DIR"
            fi
        fi

        if [ ! -d "$LLAMA_SRC_DIR" ]; then
            log_info "No existing llama.cpp found. Performing fresh clone..."
            git clone https://github.com/ggml-org/llama.cpp.git "$LLAMA_SRC_DIR"
        fi
    fi

    if ! build_llama "$LLAMA_SRC_DIR" "$BUILD_TYPE"; then
        log_error "llama.cpp build failed"
        exit 1
    fi

    if [ -f "$LLAMA_SRC_DIR/build/bin/llama-server" ] || \
       [ -f "$LLAMA_SRC_DIR/build/Release/bin/llama-server" ] || \
       [ -f "$LLAMA_SRC_DIR/llama-server" ] || \
       [ -f "$LLAMA_SRC_DIR/bin/llama-server" ]; then
        log_success "llama.cpp build completed successfully!"
    else
        log_error "Build finished but llama-server not found."
        exit 1
    fi
    # Show usage instructions only in verbose mode
    if [[ "${KAMIWAZA_DEBUG}" == "true" || -f kamiwaza-shibboleth ]]; then
        echo
        log_header "Usage Instructions"
        echo
        case "$BUILD_TYPE" in
            "sycl")
                log_info "For SYCL build (Intel GPU):"
                log_info "  1. Source oneAPI: source /opt/intel/oneapi/setvars.sh"
                log_info "  2. Run inference: $LLAMA_SRC_DIR/build/bin/llama-cli -m <model.gguf> -p 'Hello' -ngl 999"
                log_info "  3. Run benchmark: $LLAMA_SRC_DIR/build/bin/llama-bench -m <model.gguf> -ngl 999"
                ;;
            "standard")
                log_info "For standard build:"
                log_info "  1. Run server: $LLAMA_SRC_DIR/bin/llama-server -m <model.gguf>"
                ;;
        esac
        echo
    fi
}
# Run main function
main "$@"
