#!/bin/bash
# add_node_to_etcd_cluster.sh - Place in KAMIWAZA_ROOT
set -euo pipefail

RETRY_INTERVAL=20
MAX_RETRIES=30

add_node_to_cluster() {
    local new_node_name="$1"
    local container_name="${KAMIWAZA_ENV:-default}_kamiwaza-etcd-$(hostname)"
    local attempt=1

    while [ $attempt -le $MAX_RETRIES ]; do
        echo "Attempting to add node $new_node_name to cluster (attempt $attempt/$MAX_RETRIES)..."
        
        # Check if container is running
        if ! docker ps --format '{{.Names}}' | grep -q "^${container_name}$"; then
            echo "Head node etcd container not running yet..."
            sleep $RETRY_INTERVAL
            ((attempt++))
            continue
        fi

        # Check if node is already in cluster by checking member list
        if ! member_list=$(docker exec "$container_name" etcdctl \
            --cert="/etc/etcd/certs/peer-$(hostname).pem" \
            --key="/etc/etcd/certs/peer-$(hostname)-key.pem" \
            --cacert=/etc/etcd/certs/ca.pem \
            member list -w table 2>&1); then
            echo "ERROR: Failed to get member list from etcd cluster"
            echo "Member list command output: $member_list"
            exit 1
        fi

        # Look for the node name in the member list output
        if echo "$member_list" | grep -q "$new_node_name"; then
            # Node is already in cluster
            echo "SUCCESS: Node $new_node_name is already in the cluster"
            echo "$member_list"  # Display current cluster state
            return 0
        fi

        # Try to add the member
        if output=$(docker exec "$container_name" etcdctl member add \
            "$new_node_name" \
            --peer-urls="https://${new_node_name}:2380" \
            --cacert=/etc/etcd/certs/ca.pem \
            --cert="/etc/etcd/certs/peer-$(hostname).pem" \
            --key="/etc/etcd/certs/peer-$(hostname)-key.pem" 2>&1); then
            
            echo "SUCCESS: Node $new_node_name successfully added to cluster"
            return 0
        else
            # Check if it failed because member already exists
            if echo "$output" | grep -q "Peer URLs already exists"; then
                echo "SUCCESS: Node $new_node_name is already a cluster member"
                return 0
            fi
            
            echo "$output"  # Print the error for debugging
            echo "Failed to add node, retrying in $RETRY_INTERVAL seconds..."
            sleep $RETRY_INTERVAL
            ((attempt++))
        fi
    done

    echo "ERROR: Failed to add node after $MAX_RETRIES attempts"
    return 1
}

# Execute if arguments provided
if [ "$#" -eq 1 ]; then
    add_node_to_cluster "$1"
    exit $?
fi